/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "MAlignment.h"

namespace GB2 {

MAlignment::MAlignment(const QString& _name, DNAAlphabet* al, const QList<MAlignmentItem>& aitems) 
: alphabet(al), alignedSeqs(aitems)
{
    assert(al==NULL || !_name.isEmpty()); //if not default constructor -> name must be provided
    setName( _name );
    normalizeModel();
}

void MAlignment::normalizeModel() {
    //check that all sequences are of equal size
    int maxSize = 0;
    for (int i=0, n = alignedSeqs.size();  i < n; i++) {
        const MAlignmentItem& item = alignedSeqs[i];
        maxSize = qMax(item.sequence.size(), maxSize);
    }

    for (int i=0, n = alignedSeqs.size();  i < n; i++) {
        MAlignmentItem& item = alignedSeqs[i];
        int dSize = maxSize - item.sequence.size();
        if (dSize > 0) {
            item.sequence.append(QByteArray(dSize, MAlignment_GapChar));
        }
    }
}


bool MAlignment::isNormalized() const {
    if (alphabet == NULL) {
        return false;
    }
    //check that all sequences are of equal size
    int size = 0;
    for (int i=0, n = alignedSeqs.size();  i < n; i++) {
        const MAlignmentItem& item = alignedSeqs[i];
        if (i == 0) {
            size = item.sequence.size();
        } else {
            int itemSize = item.sequence.size();
            if (size!=itemSize) {
                return false;
            }
        }
    }

    //todo: check alphabet too

    return true;
}

bool MAlignment::hasGaps() const {
    for (int i=0, n = alignedSeqs.size();  i < n; i++) {
        const MAlignmentItem& item = alignedSeqs[i];
        if (item.sequence.indexOf(MAlignment_GapChar)!=-1) {
            return true;
        }
    }
    return false;
}

char MAlignment::getBase(int seqNum, int pos) const {
    assert(seqNum >= 0 && seqNum < getNumSequences());
    assert(pos >=0 && pos < getLength());
    const MAlignmentItem& item = alignedSeqs[seqNum];
    return item.sequence[pos];
}


void MAlignment::clear() {
    alphabet = NULL;
    alignedSeqs.clear();
}


MAlignment MAlignment::subAlignment(int start, int len) const {
    assert(start >=0 && start + len <= getLength());
    assert(isNormalized());

    MAlignment res;
    res.alphabet = alphabet;
    foreach(const MAlignmentItem& item, alignedSeqs) {
        res.alignedSeqs.append(MAlignmentItem(item.name, item.sequence.mid(start, len)));
    }
    return res;
}


MAlignment& MAlignment::operator+=(const MAlignment& ma) {
    assert(ma.alphabet == alphabet);
    assert(isNormalized());
    assert(ma.isNormalized());

    int nSeq = getNumSequences();
    assert(ma.getNumSequences() == nSeq);
    
    for (int i=0; i<nSeq; i++) {
        MAlignmentItem& item = alignedSeqs[i];
        item.sequence.append(ma.alignedSeqs[i].sequence);
    }
    return *this;
}

int MAlignment::estimateMemorySize() const {
    int len = getLength();
    int nSeq = getNumSequences();
    int res = len * nSeq + 100 * nSeq;
    return res;
}

}//namespace
