/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "RawDNASequenceFormat.h"

#include "DocumentFormatUtils.h"

#include <core_api/Task.h>
#include <core_api/IOAdapter.h>
#include <gobjects/GObjectTypes.h>
#include <util_text/TextUtils.h>
#include <datatype/DNASequence.h>

#include <memory>

namespace GB2 {

/* TRANSLATOR GB2::RawDNASequenceFormat */
/* TRANSLATOR GB2::IOAdapter */

RawDNASequenceFormat::RawDNASequenceFormat(QObject* p) : DocumentFormat(p)
{
	formatName = tr("raw_dna_sequence_format");
}

QStringList RawDNASequenceFormat::getSupportedDocumentFileExtensions() {
	QStringList l;
    l<<"seq" << "txt";
	return l;
}

static void load(IOAdapter* io, QList<GObject*>& objects, TaskStateInfo& ti) {
    static int READ_BUFF_SIZE = 4096;

    QByteArray readBuffer(READ_BUFF_SIZE, '\0');
    char* buff  = readBuffer.data();

    int wholeSize = io->left();
    const QBitArray& ALPHAS = TextUtils::ALPHA_NUMS;

    QByteArray seq;
    if (wholeSize != -1) {
        seq.reserve(wholeSize);
    }

    //reading sequence
    QBuffer writer(&seq);
    writer.open( QIODevice::WriteOnly | QIODevice::Append );
    bool ok = true;
    int len = 0;
    while (ok && (len = io->readBlock(buff, READ_BUFF_SIZE)) > 0) {
        if (ti.cancelFlag) {
            break;
        }
        for (int i=0; i<len && ok; i++) {
            char c = buff[i];
            if (ALPHAS[(uchar)c]) {
                ok = writer.putChar(c);
            }
        }
		ti.progress = io->getProgress();
    }
    writer.close();
    if (ti.hasErrors()) {
        return;
    }
    if (seq.size() == 0) {
        ti.setError(RawDNASequenceFormat::tr("sequence_is_empty"));
        return;
    }
    DNASequence dnaseq(seq);
    DocumentFormatUtils::addSequenceObject(objects, "Sequence", dnaseq);
}

Document* RawDNASequenceFormat::loadExistingDocument(IOAdapterFactory* iof, const QString& url, TaskStateInfo& ti, const QVariantMap& fs) {
	std::auto_ptr<IOAdapter> io(iof->createIOAdapter());

	if (!io->open(url, IOAdapterMode_Read)) {
		ti.setError(IOAdapter::tr("error_opening_url_for_read '%1'").arg(url));
		return NULL;
	}
    QList<GObject*> objects;
    load(io.get(), objects, ti);
    io->close();

    if (ti.hasErrors()) {
        return NULL;
    }

    Document* doc = new Document(this, iof, url, objects, fs);
    return doc;
}

bool RawDNASequenceFormat::isDataFormatSupported(const char* data, int size) const {
	return !TextUtils::contains(TextUtils::BINARY, data, size);
}

bool RawDNASequenceFormat::isObjectOpSupported(const Document* d , DocumentFormat::DocObjectOp op, GObjectType t) const{
    Q_UNUSED(d); Q_UNUSED(op);
	return (t == GObjectTypes::DNA_SEQUENCE);
}

bool RawDNASequenceFormat::checkConstraints(const DocumentFormatConstraints& c) const {
	foreach (GObjectType t, c.supportedObjectTypes) {
		if (t!=GObjectTypes::DNA_SEQUENCE) {
			return false;
		}
	}
	if (c.checkRawData) {
		return isDataFormatSupported(c.rawData.constData(), c.rawData.size());
	}
    if (c.mustSupportWrite) {
        return false;
    }
    if( c.supportsStreamingRead ) {
        return false;
    }
    return true;
}


}//namespace
