/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "TestRunnerTask.h"

#include <core_api/AppContext.h>
#include <core_api/Log.h>
#include <core_api/Timer.h>
#include <test_framework/GTestFrameworkComponents.h>
#include <test_framework/GTest.h>
#include <QtCore/QtAlgorithms>

namespace GB2
{
#define ULOG_CAT_PLUGIN_TEST_RUNNER "Test Runner"
static LogCategory log(ULOG_CAT_PLUGIN_TEST_RUNNER);

TestRunnerTask::TestRunnerTask(const QList<GTestState*>& tests, const GTestEnvironment* _env, int testSizeToRun)
: Task( tr("test_runner"), TaskFlag_NoRun),  env(_env)
{
    tpm = Task::Progress_SubTasksBased;
    setMaxParallelSubtasks(testSizeToRun);

    sizeToRun = testSizeToRun;
    assert(!tests.isEmpty());
    awaitingTests=tests;
    foreach(GTestState *t,awaitingTests){
        t->clearState();
    }
    for (int i=0; !awaitingTests.isEmpty() && i < sizeToRun; i++){
        GTestState *t=awaitingTests.takeFirst();
        LoadTestTask* lt = new LoadTestTask(t);
        lt->setSubtaskProgressWeight(0);
        addSubTask(lt);
    }
}


QList<Task*> TestRunnerTask::onSubTaskFinished(Task* subTask) {
    stateInfo.progress = 100*getSubtasks().size()/(getSubtasks().size() + 2*awaitingTests.size());
    QList<Task*> res;
    if(isCanceled()){
        return res;
    }
    LoadTestTask* loader = qobject_cast<LoadTestTask*>(subTask);
    if (loader == NULL) {
        GTest* test = qobject_cast<GTest*>(subTask);
        assert(test);
        test->cleanup();
        GTestState* testState = stateByTest.value(test);
        assert(testState!=NULL);
        assert(testState->isNew());
        if (test->hasErrors()) {
            testState->setFailed(test->getStateInfo().getError());
            log.info(QString("##teamcity[testStarted name='%1 : %2']").arg(testState->getTestRef()->getSuite()->getName(),testState->getTestRef()->getShortName()));
            log.info(QString("##teamcity[testFailed name='%1 : %2' message='%3' details='%3']").arg(testState->getTestRef()->getSuite()->getName(),testState->getTestRef()->getShortName(),QString(testState->getErrorMessage()).replace("'","|'").replace('\n',' ')));
            log.info(QString("##teamcity[testFinished name='%1 : %2']").arg(testState->getTestRef()->getSuite()->getName(),testState->getTestRef()->getShortName()));
        } else {
            testState->setPassed();
            log.info(QString("##teamcity[testStarted name='%1 : %2']").arg(testState->getTestRef()->getSuite()->getName(),testState->getTestRef()->getShortName()));
            log.info(QString("##teamcity[testFinished name='%1 : %2' duration='%3']").arg(testState->getTestRef()->getSuite()->getName(), testState->getTestRef()->getShortName(),QString::number(GTimer::millisBetween(test->getTimeInfo().startTime, test->getTimeInfo().finishTime))));
        }
        if(!awaitingTests.isEmpty()){
            GTestState *t=awaitingTests.takeFirst();
            res.append(new LoadTestTask(t));
        }
    } else {
        GTestState* testState = loader->testState;
        if (loader->hasErrors()) {
            testState->setFailed(loader->getStateInfo().getError());
            log.info(QString("##teamcity[testStarted name='%1 : %2']").arg(testState->getTestRef()->getSuite()->getName(),testState->getTestRef()->getShortName()));
            log.info(QString("##teamcity[testFailed name='%1 : %2' message='%3' details='%3']").arg(testState->getTestRef()->getSuite()->getName(),testState->getTestRef()->getShortName(),QString(testState->getErrorMessage()).replace("'","|'").replace('\n',' ')));
            log.info(QString("##teamcity[testFinished name='%1 : %2']").arg(testState->getTestRef()->getSuite()->getName(),testState->getTestRef()->getShortName()));
        }  else  {
            GTestFormatId id = testState->getTestRef()->getFormatId();
            GTestFormat* tf = AppContext::getTestFramework()->getTestFormatRegistry()->findFormat(id);
            if (tf == NULL) {
                testState->setFailed(tr("test_format_not_supported_%1").arg(id));
                log.info(QString("##teamcity[testStarted name='%1 : %2']").arg(testState->getTestRef()->getSuite()->getName(),testState->getTestRef()->getShortName()));
                log.info(QString("##teamcity[testFailed name='%1 : %2' message='%3' details='%3']").arg(testState->getTestRef()->getSuite()->getName(),testState->getTestRef()->getShortName(),QString(testState->getErrorMessage()).replace("'","|'").replace('\n',' ').replace('\n',' ')));
                log.info(QString("##teamcity[testFinished name='%1 : %2']").arg(testState->getTestRef()->getSuite()->getName(),testState->getTestRef()->getShortName()));
            } else {
                QString err;
                GTestEnvironment * newEnv = new GTestEnvironment();
                //caching newly created environment. will be deleted in cleanup()
                mergedSuites.push_back( newEnv );
                GTestEnvironment * testParentEnv = testState->getTestRef()->getSuite()->getEnv();
                QMap<QString, QString> parentVars = testParentEnv->getVars();
                QList<QString> parentVarsNames = parentVars.keys();
                foreach( const QString & parentVar, parentVarsNames ) {
                    newEnv->setVar( parentVar, parentVars[parentVar] );
                }

                //overriding existing variables with global ones
                QMap<QString, QString> globalEnvVars = env->getVars();
                foreach( QString var, globalEnvVars.keys() ) {
                    newEnv->setVar( var, globalEnvVars[var] );
                }

                GTest* test = tf->createTest(testState->getTestRef()->getShortName(), NULL, newEnv, loader->testData, err);
                if (test == NULL) {
                    testState->setFailed(err);
                    log.info(QString("##teamcity[testStarted name='%1 : %2']").arg(testState->getTestRef()->getSuite()->getName(),testState->getTestRef()->getShortName()));
                    log.info(QString("##teamcity[testFailed name='%1 : %2' message='%3' details='%3']").arg(testState->getTestRef()->getSuite()->getName(),testState->getTestRef()->getShortName(),QString(testState->getErrorMessage()).replace("'","|'").replace('\n',' ')));
                    log.info(QString("##teamcity[testFinished name='%1 : %2']").arg(testState->getTestRef()->getSuite()->getName(),testState->getTestRef()->getShortName()));
                } else {
                    stateByTest[test] = testState;
                    QString var = env->getVar(TIME_OUT_VAR);
                    if (var!="0") {
                        test->setTimeOut(var.toInt());
                    }
                    res.append(test);
                }
            }
        }
    }
    return res;
}

void TestRunnerTask::cleanup() {
    qDeleteAll( mergedSuites );
    Task::cleanup();
}


LoadTestTask::LoadTestTask(GTestState* _testState)
: Task( tr("test_loader_for_%1").arg(_testState->getTestRef()->getShortName()), TaskFlag_None), testState(_testState)
{
}

void LoadTestTask::run() {
    const QString& url = testState->getTestRef()->getURL();
    QFile f(url);
    if (!f.open(QIODevice::ReadOnly)) {
        stateInfo.setError(  tr("cant_open_file: %1").arg(url) );
        return;
    }
    testData = f.readAll();
    f.close();
}


} //namespace
