/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "ADVGraphModel.h"
#include "GSequenceGraphView.h"
#include "WindowStepSelectorWidget.h"

#include <math.h>

#include <core_api/Log.h>
#include <time.h>

namespace GB2 {

GSequenceGraphData::GSequenceGraphData(const QString& _graphName) : graphName(_graphName), ga(NULL)
{
    cachedFrom = cachedLen = cachedW = cachedS = 0;;
}

GSequenceGraphData::~GSequenceGraphData() {
	if (ga!=NULL) {
		delete ga;
	}
}

void GSequenceGraphUtils::calculateMinMax(const QVector<float>& data, float& min, float& max)  {
	assert(data.size() > 0);
	min = max = data.first();
	const float* d = data.data();
	for (int i=1, n = data.size() ; i<n ; i++) {
		float val = d[i];
		if (min > val) {
			min = val;
		} else if (max < val) {
			max = val;
		}
	}
}

#define ACCEPTABLE_FLOAT_PRESISION_LOSS 0.0001
float GSequenceGraphUtils::calculateAverage(const QVector<float>& data, float start, float range) {
	float result;
	if (int(start)!=int(start+range)) {
		//result constructed from 3 parts: ave[start, startIdx] + ave[startIdx, endIdx] + ave[endIdx, end]
		float part1 = 0;
		float part2 = 0;
		float part3 = 0;

		int startIdx = int(floor(start));
		float startDiff = 1 - (start - startIdx);
		float end = start + range;
		int endIdx = int(end);
		float endDiff = end - endIdx;

		assert(qAbs(startDiff + (endIdx-(startIdx+1)) + endDiff - range) / range <= ACCEPTABLE_FLOAT_PRESISION_LOSS);

		//calculating part1
		if (startDiff > ACCEPTABLE_FLOAT_PRESISION_LOSS) {
			float v1 = data[startIdx];
			float v2 = data[startIdx+1];
			float k = v2-v1;
			float valInStart = v2 - k*startDiff;
			part1 = startDiff * (valInStart + v2) / 2;
		}
		int firstIdxInRange = int(ceil(start));
		//calculating part2
		for(int i =firstIdxInRange; i < endIdx; i++) {
			part2+=data[i];
		}
		//calculating part3
		if (endDiff > ACCEPTABLE_FLOAT_PRESISION_LOSS && endIdx+1 < (int)data.size()) {
			float v1 = data[endIdx];
			float v2 = data[endIdx+1];
			float k = v2-v1;
			float valInEnd= v1+k*endDiff;
			part3 = endDiff * (v1 + valInEnd) / 2;
		}
		//sum
		result = (part1 + part2 + part3 ) / range;
	} else {
		//result constructed from 1 part: ave[start, end], no data points between
		int startIdx = int(start);
		float startDiff = start - float(startIdx);
		float endDiff = startDiff + range;
		assert(endDiff < 1);
		float v1 = data[startIdx];
		float v2 = data[startIdx+1];
		float k = v2-v1;
		float valInStart = v1 + k*startDiff;
		float valInEnd = v1 + k*endDiff;
		result = (valInEnd+valInStart)/2;
	}
	return result;
}

void GSequenceGraphUtils::fitToScreen(const QVector<float>& data, int dataStartBase, int dataEndBase, QVector<float>& results, 
									   int resultStartBase, int resultEndBase, int screenWidth, float unknownVal) 
{
	//BUG:422: use intervals and max/min values instead of average!
	float basesPerPixel = (resultEndBase - resultStartBase) / (float) screenWidth;
	float basesInDataPerIndex = (dataEndBase - dataStartBase) / (float) (data.size() - 1);
	float currentBase = resultStartBase;
	results.reserve(results.size() + screenWidth);
	for (int i=0; i < screenWidth; i++, currentBase+=basesPerPixel) {
		float dataStartIdx = (currentBase - basesPerPixel / 2 - dataStartBase) / basesInDataPerIndex;
		float dataEndIdx =  (currentBase  + basesPerPixel / 2 - dataStartBase) / basesInDataPerIndex;
		dataStartIdx = qMax((float)0, dataStartIdx);
		dataEndIdx = qMin((float)data.size()-1, dataEndIdx);
		float nDataPointsToAverage = dataEndIdx - dataStartIdx;
		float val = unknownVal;
		if (nDataPointsToAverage >= ACCEPTABLE_FLOAT_PRESISION_LOSS) {
			val = calculateAverage(data, dataStartIdx, nDataPointsToAverage);
		}
		results.append(val);
	}
}

int GSequenceGraphUtils::getNumSteps(const LRegion& range, int w, int s) { 
	assert(range.len >= w);
	int steps = (range.len  - w) / s + 1;
	return steps;
}

//////////////////////////////////////////////////////////////////////////
//drawer

GSequenceGraphDrawer::GSequenceGraphDrawer(GSequenceGraphView* v, const GSequenceGraphWindowData& wd) 
: QObject(v), view(v), wdata(wd) 
{
	defFont = new QFont("Arial", 8);
}

GSequenceGraphDrawer::~GSequenceGraphDrawer() {
	delete defFont;
}

//TODO:
#define UNKNOWN_VAL -1 
static LogCategory logPerf(ULOG_CAT_PERFORMANCE);

void GSequenceGraphDrawer::draw(QPainter& p, GSequenceGraphData* d, const QRect& rect) {
	float min=0;
	float max=0;
	PairVector points;

	calculatePoints(d, points, min, max, rect.width());

	assert(points.firstPoints.size() == rect.width());

	double comin = commdata.min, comax = commdata.max;
	QString minprefix, maxprefix;
	if (commdata.e)	{
		min = comin; 
		max = comax;
		minprefix = "<=";
		maxprefix = ">=";
	}

    {
		//draw min/max
		QPen minMaxPen(Qt::DashDotDotLine);
		minMaxPen.setWidth(1);
		p.setPen(minMaxPen);
		p.setFont(*defFont);

		//max
		p.drawLine(rect.topLeft(), rect.topRight()); 
		QRect maxTextRect(rect.x(), rect.y(), rect.width(), 12);
		p.drawText(maxTextRect, Qt::AlignRight, maxprefix + QString::number((double) max, 'g', 4));

		//min 
		p.drawLine(rect.bottomLeft(), rect.bottomRight());
		QRect minTextRect(rect.x(), rect.bottom()-12, rect.width(), 12);
		p.drawText(minTextRect, Qt::AlignRight, minprefix + QString::number((double) min, 'g', 4));
	}

	QPen graphPen(Qt::SolidLine);
	graphPen.setWidth(1);
	p.setPen(graphPen);


	int graphHeight = rect.bottom() - rect.top() - 2;
	float kh = (min == max) ? 1 : graphHeight / (max - min);

	int prevY = -1;
	int prevX = -1;


	if (!commdata.e)	{
		////////cutoff off
		for (int i=0, n = points.firstPoints.size(); i < n; i++) {
			float fy1 = points.firstPoints[i];
			if (fy1 == UNKNOWN_VAL) {
				continue;
			}
			int dy1 = qRound((fy1 - min) * kh);
			assert(dy1 <= graphHeight);
			int y1 = rect.bottom() - 1 - dy1;
			int x = rect.left() + i;
			assert(y1 > rect.top() && y1 < rect.bottom());
			if (prevX!=-1){
				p.drawLine(prevX, prevY , x, y1);
			} 
			prevY = y1;
			prevX = x;
			if (points.useIntervals)	{
				float fy2 = points.secondPoints[i];
				if (fy2 == UNKNOWN_VAL) {
					continue;
				}
				int dy2 = qRound((fy2 - min) * kh);
				assert(dy2 <= graphHeight);
				int y2 = rect.bottom() - 1 - dy2;
				assert(y2 > rect.top() && y2 < rect.bottom());
				if (prevX!=-1){
					p.drawLine(prevX, prevY , x, y2);
				}	 
				prevY = y2;
				prevX = x;
			}
		}
	} else	{
		////////cutoff on
		
		float fymin = comin;
		float fymax = comax;
		float fymid = (comin + comax)/2;
		float fy;
		int prevFY = -1;
		bool rp = false, lp = false;
		int ymid = rect.bottom() - 1 - qRound((fymid - min) * kh);
		if (points.useIntervals)	{
			for (int i=0, n = points.firstPoints.size(); i < n; i++) {
				fy = points.firstPoints[i];
				if (fy == UNKNOWN_VAL) {
					continue;
				}
				if (fy >= fymax) {
					fy = fymax;
				} 
				else {
					if (fy <= fymin) fy = fymin; 
							else	fy = fymid;
				}
				int dy = qRound((fy - min) * kh);
				assert(dy <= graphHeight);
				int y = rect.bottom() - 1 - dy;
				int x = rect.left() + i;
				assert(y > rect.top() && y < rect.bottom());
				p.drawLine(x, ymid , x, y);
			}
			for (int i=0, n = points.secondPoints.size(); i < n; i++) {
				fy = points.secondPoints[i];
				if (fy == UNKNOWN_VAL) {
					continue;
				}
				if (fy >= fymax) {
					fy = fymax;
				} 
				else {
					if (fy <= fymin) fy = fymin; 
							else	fy = fymid;
				}
				int dy = qRound((fy - min) * kh);
				assert(dy <= graphHeight);
				int y = rect.bottom() - 1 - dy;
				int x = rect.left() + i;
				assert(y > rect.top() && y < rect.bottom());
				p.drawLine(x, ymid , x, y);
			}
		} else	{
			for (int i=0, n = points.firstPoints.size(); i < n; i++) {
				fy = points.firstPoints[i];
				rp = false;
				lp = false;
					if (fy == UNKNOWN_VAL) {
					continue;
				}
				if (fy >= fymax) {
					fy = fymax;
					if (prevFY == int(fymid)) lp=true;
				} 
				else {
					fy = fymid;
					if (prevFY == int(fymax)) rp=true;
				}
				int dy = qRound((fy - min) * kh);
				assert(dy <= graphHeight);
				int y = rect.bottom() - 1 - dy;
				int x = rect.left() + i;
				if (lp) {
					p.drawLine(prevX, prevY, x, prevY);
					prevX = x;
				}
				if (rp) {
					p.drawLine(prevX,prevY,prevX,y);
					prevY = y;
				}
				assert(y > rect.top() && y < rect.bottom());
				if (prevX!=-1){
					p.drawLine(prevX, prevY , x, y);
				} 
				prevY = y;
				prevX = x;
                                prevFY = (int) fy;
			}
			prevY = -1;
			prevX = -1;
			prevFY = -1;
			for (int i=0, n = points.firstPoints.size(); i < n; i++) {
				fy = points.firstPoints[i];
				rp = false;
				lp = false;
				if (fy == UNKNOWN_VAL) {
					continue;
				}
				if (fy <= fymin) {
					fy = fymin;
					if (prevFY == int(fymid)) lp=true;
				} 
				else {
					fy = fymid;
					if (prevFY == int(fymin)) rp=true;
				}
				int dy = qRound((fy - min) * kh);
				assert(dy <= graphHeight);
				int y = rect.bottom() - 1 - dy;
				int x = rect.left() + i;
				if (lp) {
					p.drawLine(prevX, prevY, x, prevY);
					prevX = x;
				}
				if (rp) {
					p.drawLine(prevX,prevY,prevX,y);
					prevY = y;
				}
				assert(y > rect.top() && y < rect.bottom());
				if (prevX!=-1){
					p.drawLine(prevX, prevY , x, y);
				} 
				prevY = y;
				prevX = x;
                                prevFY = (int) fy;
			}
		}
	}
}

static void align(int start, int end, int win, int step, int seqLen, int& alignedFirst, int& alignedLast) {
	int win2 = (win + 1) / 2;
	int notAlignedFirst = start - win2;
	alignedFirst = qMax(0, notAlignedFirst - notAlignedFirst % step);
	
	int notAlignedLast = end + win + step;
	alignedLast = notAlignedLast - notAlignedLast % step;
	while (alignedLast + win2 >= end + step) {
		alignedLast-=step;
	}
	while (alignedLast > seqLen - win) {
		alignedLast-=step;
	}
	assert(alignedLast % step == 0);
	assert(alignedFirst % step == 0);
	assert(alignedLast < end);
}




void GSequenceGraphDrawer::calculatePoints(GSequenceGraphData* d, PairVector& points, float& min, float& max, int numPoints) {
    const LRegion& vr = view->getVisibleRange();
	int step = wdata.step;
	int win = wdata.window;
	int seqLen = view->getSequenceLen();

	points.firstPoints.resize(numPoints);
	points.firstPoints.fill(UNKNOWN_VAL);
	points.secondPoints.resize(numPoints);
	points.secondPoints.fill(UNKNOWN_VAL);
	min = UNKNOWN_VAL;
	max = UNKNOWN_VAL;
	if (vr.len < win) {
		return;
	}
	int alignedFirst = 0; //start point for the first window
	int alignedLast = 0; //start point for the last window
	align(vr.startPos, vr.endPos(), win, step, seqLen, alignedFirst, alignedLast);
	int nSteps = (alignedLast - alignedFirst) / step;
    
	bool winStepNotChanged = win == d->cachedW && step == d->cachedS ;
    bool numPointsNotChanged = numPoints == d->cachedData.firstPoints.size();
	
    bool useCached = vr.len == d->cachedLen && vr.startPos == d->cachedFrom 
		&& winStepNotChanged && numPointsNotChanged;
    
    if (useCached) {
        points = d->cachedData;
    } else if (nSteps > numPoints) {
		points.useIntervals = true;
		int stepsPerPoint = nSteps / points.firstPoints.size();
		int basesPerPoint = stepsPerPoint * step;

		//<=step because of boundary conditions -> num steps can be changed if alignedLast+w2 == end
		bool offsetIsTooSmall = qAbs((d->alignedLC - d->alignedFC) - (alignedLast - alignedFirst)) <= step 
						&& (qAbs(alignedFirst - d->alignedFC) < basesPerPoint);

		if (offsetIsTooSmall && winStepNotChanged && numPointsNotChanged && vr.len == d->cachedLen ) {
			useCached = true;
			points = d->cachedData;
		} else {
            clock_t start = clock();
			calculateWithFit(d, points, alignedFirst, alignedLast);
            QString t = QString::number((clock() - start)/(double)CLOCKS_PER_SEC);
            logPerf.trace(QString("graph '%1' calculation time %2").arg(d->graphName).arg(t));
		}
	} else {
		points.useIntervals = false;
		calculateWithExpand(d, points, alignedFirst, alignedLast);
	}

    bool inited = false; min = 0; max = 0;
	foreach(float p, points.firstPoints) {
		if (p == UNKNOWN_VAL) {
			continue;
		}
		if (!inited) {
			inited = true;
			min = p;
			max = p;
		} else {
			min = qMin(p, min);
			max = qMax(p, max);
		}
	}
	if (points.useIntervals)	{
		foreach(float p, points.secondPoints) {
			if (p == UNKNOWN_VAL) {
				continue;
			}
			min = qMin(p, min);
			max = qMax(p, max);

		}
	}
	if (useCached) {
		return;
	}
    d->cachedData = points;
    d->cachedFrom = vr.startPos;
    d->cachedLen = vr.len;
    d->cachedW = win;
    d->cachedS = step;
	d->alignedFC = alignedFirst;
	d->alignedLC = alignedLast;
}

void GSequenceGraphDrawer::calculateWithFit(GSequenceGraphData* d, PairVector& points, int alignedFirst, int alignedLast) {
	int win = wdata.window;
	int step  = wdata.step;

	int nSteps = (alignedLast - alignedFirst)/step;
	int lastBase = alignedLast + win;
    
	int stepsPerPoint = nSteps / points.firstPoints.size();
	assert(stepsPerPoint >= 1);
	int basesPerPoint = stepsPerPoint*step;
	QVector<float> pointData;
	DNASequenceObject* o = view->getSequenceObject();
	int len = qMax(int(basesPerPoint), win);
	for (int i=0; i < points.firstPoints.size(); i++) {
		pointData.clear();
        LRegion r(int(alignedFirst + i*basesPerPoint), len);
		assert(r.endPos() <= lastBase);
		d->ga->calculate(pointData, o, r, &wdata);
		float min, max;
		GSequenceGraphUtils::calculateMinMax(pointData, min, max);

		points.firstPoints[i] = max; //BUG:422: support interval based graph!!!
		points.secondPoints[i] = min;
	}
}

void GSequenceGraphDrawer::calculateWithExpand(GSequenceGraphData* d, PairVector& points, int alignedFirst, int alignedLast) {
	int win = wdata.window;
	int win2 = (win+1)/2;
	int step = wdata.step;
	assert((alignedLast - alignedFirst) % step == 0);

	LRegion r(alignedFirst, alignedLast - alignedFirst + win);
	QVector<float> res;
	d->ga->calculate(res, view->getSequenceObject(), r, &wdata);
	const LRegion& vr = view->getVisibleRange();
	
	assert(alignedFirst + win2 + step >= vr.startPos); //0 or 1 step is before the visible range
	assert(alignedLast + win2 - step <= vr.endPos()); //0 or 1 step is after the the visible range 

	bool hasBeforeStep = alignedFirst + win2 < vr.startPos;
	bool hasAfterStep  = alignedLast + win2 >= vr.endPos();

	int firstBaseOffset = hasBeforeStep ? 
		(step - (vr.startPos - (alignedFirst + win2)))
		: (alignedFirst + win2 - vr.startPos);
	int lastBaseOffset = hasAfterStep ? 
		(step - (alignedLast + win2 - vr.endPos()))  //extra step on the right is available
		: (vr.endPos() - (alignedLast + win2)); // no extra step available -> end of the sequence
	
	assert(firstBaseOffset >= 0 && lastBaseOffset >= 0); 
	assert(hasBeforeStep ? (firstBaseOffset < step && firstBaseOffset!=0): firstBaseOffset <= win2);
	assert(hasAfterStep ? (lastBaseOffset <= step && lastBaseOffset !=0) : lastBaseOffset < win2 + step);
	
	float base2point = points.firstPoints.size() / (float)vr.len;

	int ri = hasBeforeStep ? 1 : 0;
	int rn = hasAfterStep ? res.size()-1 : res.size();
	for (int i=0;  ri < rn; ri++, i++) {
		int b = firstBaseOffset + i * step;
		int px = int(b * base2point);
		assert(px < points.firstPoints.size());
		points.firstPoints[px] = res[ri];
	}

	//restore boundary points if possible
	if (hasBeforeStep && res[0]!=UNKNOWN_VAL && res[1]!=UNKNOWN_VAL) {
		assert(firstBaseOffset > 0);
		float k = firstBaseOffset / (float)step;
		float val = res[1] + (res[0]-res[1])*k;
		points.firstPoints[0] = val;
	}

	if (hasAfterStep && res[rn-1]!=UNKNOWN_VAL && res[rn]!=UNKNOWN_VAL) {
		assert(lastBaseOffset > 0);
		float k = lastBaseOffset / (float)step;
		float val = res[rn-1] + (res[rn]-res[rn-1])*k;
		points.firstPoints[points.firstPoints.size()-1] = val;
	}
}


void GSequenceGraphDrawer::showSettingsDialog() {
	WindowStepSelectorDialog d(view, LRegion(1, view->getSequenceLen()-1), wdata.window, wdata.step, commdata.min, commdata.max, commdata.e);
    int res = d.exec();
    if (res == QDialog::Accepted) {
        wdata.window = d.getWindowStepSelector()->getWindow();
        wdata.step = d.getWindowStepSelector()->getStep();
		commdata.e = d.getMinMaxSelector()->getState();
		commdata.min = d.getMinMaxSelector()->getMin();
		commdata.max = d.getMinMaxSelector()->getMax();
        view->update();
    }
}


} // namespace
