/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "IntegralBus.h"
#include "IntegralBusType.h"

#include <core_api/Log.h>
#include <limits.h>

namespace GB2 {
namespace Workflow {

static LogCategory log(ULOG_CAT_WD);

static QMap<QString, QStringList> getListMappings(const QStrStrMap& bm, const Port* p) {
    assert(p->isInput());    
    DataTypePtr dt = p->getType();
    QMap<QString, QStringList> res;
    if (dt->isList()) {
        if (bm.contains(p->getId())) {
            res.insert(p->getId(), bm.value(p->getId()).split(";"));
        }
    } else if (dt->isMap()) {
        foreach(Descriptor d, dt->getElements()) {
            if (dt->getElement(d)->isList() && bm.contains(d.getId())) {
                res.insert(d.getId(), bm.value(d.getId()).split(";"));
            }
        }
    }
    return res;
}


IntegralBus::IntegralBus(Port* p) : busType(p->getType()), complement(NULL) {
    QString name = p->owner()->getLabel() + "[" + p->owner()->getId()+"]";
    if (p->isInput()) {
        Attribute* a = p->getParameter(BusPort::BUS_MAP);
        assert(a);
        if (a) {
            busMap = a->value.value<QStrStrMap>();
            assert(!busMap.isEmpty());
            QMapIterator<QString, QString> it(busMap);
            while (it.hasNext()) {
                it.next();
                log.trace(QString("%1 - input bus map key=%2 val=%3").arg(name).arg(it.key()).arg(it.value()));
            }
            listMap = getListMappings(busMap, p);
        }

    } else {
        BusPort* bp = qobject_cast<BusPort*>(p);
        DataTypePtr t = bp ? bp->getOwnType() : p->getType();
        if (t->isMap()) {
            foreach(Descriptor d, t->getElements()) {
                QString key = d.getId();
                QString val = IntegralBusType::assignSlotDesc(d, p).getId();
                busMap.insert(key, val);
            }
        } else {
            busMap.insert(p->getId(), IntegralBusType::assignSlotDesc(*p, p).getId());
        }
        QMapIterator<QString, QString> it(busMap);
        while (it.hasNext()) {
            it.next();
            log.trace(QString("%1 - output bus map key=%2 val=%3").arg(name).arg(it.key()).arg(it.value()));
        }
    }
}
    
Message IntegralBus::get() {
    QVariantMap result;
    context.clear();
    foreach (CommunicationChannel* ch, outerChannels) {
        Message m = ch->get();
        assert(m.getData().type() == QVariant::Map);
        QVariantMap imap = m.getData().toMap();
        context.unite(imap);
        foreach(QString ikey, imap.uniqueKeys()) {
            QVariant ival = imap.value(ikey);
            foreach(QString rkey, busMap.keys(ikey)) {
                log.trace("reducing bus from key="+ikey+" to="+rkey);
                result[rkey] = ival;
            }
            QMapIterator<QString,QStringList> lit(listMap);
            while (lit.hasNext())
            {
                lit.next();
                QString rkey = lit.key();
                assert(!lit.value().isEmpty());
                if (lit.value().contains(ikey)) {
                    QVariantList vl = result[rkey].toList();
                    if (m.getType()->getElement(ikey)->isList()) {
                        vl += ival.toList();
                        log.trace("reducing bus key="+ikey+" to list of "+rkey);
                    } else {
                        vl.append(ival);
                        log.trace("reducing bus key="+ikey+" to list element of "+rkey);
                    }
                    result[rkey] = vl;
                }
            }
        }
    }
    //assert(busType->isMap() || result.size() == 1);
    QVariant data;
    if (busType->isMap()) {
        data.setValue(result);
    } else if (result.size() == 1) {
        data = result.values().at(0);
    }
    if (complement) {
        complement->setContext(context);
    }
    return Message(busType, data);
}

Message IntegralBus::composeMessage(const Message& m) {
    QVariantMap data(getContext());
    if (m.getData().type() == QVariant::Map) {
        QMapIterator<QString, QVariant> it(m.getData().toMap());
        while (it.hasNext()) {
            it.next();
            QString key = busMap.value(it.key());
            log.trace("putting key="+key+" remapped from="+it.key());

            data.insert(key, it.value());
        }
    } else {
        assert(busMap.size() == 1);
        data.insert(busMap.values().first(), m.getData());
    }
    return Message(busType, data);
}

void IntegralBus::put(const Message& m) {
    Message busM = composeMessage(m);
    foreach(CommunicationChannel* ch, outerChannels) {
        ch->put(busM);
    }
}

int IntegralBus::hasMessage() const {
    if (outerChannels.isEmpty()) {
        return 0;
    }
    int num = INT_MAX;
    foreach(CommunicationChannel* ch, outerChannels) {
        num = qMin(num, ch->hasMessage());
    }
    return num;
}

int IntegralBus::hasRoom(const DataType*) const {
    if (outerChannels.isEmpty()) {
        return 0;
    }
    int num = INT_MAX;
    foreach(CommunicationChannel* ch, outerChannels) {
        num = qMin(num, ch->hasRoom());
    }
    return num;
}

bool IntegralBus::isEnded() const {
    foreach(CommunicationChannel* ch, outerChannels) {
        if (ch->isEnded()) {
#ifdef _DEBUG
            foreach(CommunicationChannel* dbg, outerChannels) {
                assert(dbg->isEnded());
            }
#endif
            return true;
        }
    }
    return false;
}

void IntegralBus::setEnded() {
    foreach(CommunicationChannel* ch, outerChannels) {
        ch->setEnded();
    }
}

}//namespace Workflow
}//namespace gb2
