/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtGui/QMouseEvent>
#include <QtGui/QMessageBox>
#include <QtGui/QColorDialog>
#include <QtGui/QFileDialog>
#include <QtGui/QImageWriter>
#include <QtCore/QTime>

#include <datatype/BioStruct3D.h>
#include <util_ov_annotated_dna/AnnotatedDNAView.h>
#include <util_ov_annotated_dna/ADVSequenceObjectContext.h>
#include <gobjects/AnnotationSettings.h>
#include <gobjects/AnnotationTableObject.h>
#include <gobjects/DNASequenceObject.h>
#include <gobjects/BioStruct3DObject.h>
#include <gobjects/GObjectUtils.h>
#include <gobjects/GObjectRelationRoles.h>
#include <core_api/Log.h>
#include <core_api/AppContext.h>
#include <core_api/DocumentModel.h>
#include <core_api/Counter.h>
#include <molecular_geometry/MolecularSurfaceFactoryRegistry.h>
#include <selection/DNASequenceSelection.h>
#include <selection/AnnotationSelection.h>
#include <time.h>

#include "BioStruct3DGLWidget.h"
#include "BioStruct3DGLRender.h"
#include "BioStruct3DColorScheme.h"
#include "GLFrameManager.h"
#include "ExportImageDialog.h"
#include "SettingsDialog.h"
#include "MolecularSurfaceRenderer.h"
#include "gl2ps/gl2ps.h"

//disable "unsafe functions" deprecation warnings on MS VS
#ifdef Q_OS_WIN
#pragma warning(disable : 4996)
#endif

namespace GB2 { 

static LogCategory log(ULOG_CAT_PLUGIN_BIOSTRUCT_3D);
static LogCategory logPerf(ULOG_CAT_PERFORMANCE);
int BioStruct3DGLWidget::widgetCount = 0;


BioStruct3DGLWidget::BioStruct3DGLWidget(const BioStruct3DObject* obj, const AnnotatedDNAView* view, GLFrameManager* manager, QWidget *parent /* = 0*/)
    : QGLWidget(parent), biostruc(obj->getBioStruct3D()), biostrucDoc(obj->getDocument()),  
    dnaView(view), frameManager(manager), molSurface(NULL), currentModelID(obj->getBioStruct3D().pdbId), spinAngle(0), displayMenu(NULL)
{
    GCOUNTER( cvar, tvar, "BioStruct3DGLWidget" );
    setObjectName(currentModelID + "-" + QString("%1").arg(++widgetCount));
    //TODO: ? setFormat(QGLFormat(QGL::DoubleBuffer | QGL::DepthBuffer));
    colorSchemeFactoryMap = BioStruct3DColorSchemeFactory::createFactories();
    rendererFactoryMap = BioStruct3DGLRendererFactory::createFactories();
    surfaceRendererFactoryMap = MolecularSurfaceRendererFactory::createFactories();
    createActiveModelIndexList();
    createActions();
    createMenus();
    connectExternalSignals();
    backgroundColor = Qt::black;
    selectionColor = Qt::yellow;
    chainIdCache.clear();
    
    // Load renderers and color schemes
    loadColorSchemes();
    loadGLRenderers();
    
    // Set view settings
    float scaleFactor = 2.5;
    cameraDistance = scaleFactor * biostruc.getMaxDistFromCenter();
    cameraClipNear = (cameraDistance - biostruc.getMaxDistFromCenter()) * 0.66f;
    cameraClipFar = (cameraDistance + biostruc.getMaxDistFromCenter()) * 1.2f; 
        
    glFrame.reset( new GLFrame(this, cameraClipNear, cameraClipFar) );
    frameManager->addGLFrame(glFrame.get());
    saveDefaultSettings();
}

BioStruct3DGLWidget::~BioStruct3DGLWidget()
{
    // Clean up factories
    
    foreach(QString key, colorSchemeFactoryMap.keys()) {
         BioStruct3DColorSchemeFactory* f = colorSchemeFactoryMap.take(key);
         delete f;
    }
    
    foreach(QString key, rendererFactoryMap.keys()) {
        BioStruct3DGLRendererFactory* f = rendererFactoryMap.take(key);
        delete f;
    }
    
    foreach(QString key, surfaceRendererFactoryMap.keys()) {
        MolecularSurfaceRendererFactory* f = surfaceRendererFactoryMap.take(key);
        delete f;
    }

    log.trace("Biostruct3DGLWdiget "+objectName()+" deleted");
}



void BioStruct3DGLWidget::initializeGL()
{
    setLightPosition(Vector3D(0, 0.0, 1.0));
    GLfloat light_diffuse[] = { 0.8f, 0.8f, 0.8f, 1.0 };
    GLfloat light_specular[] = { 0.6f, 0.6f, 0.6f, 1.0 };
    GLfloat mat_specular[] = { 0.6f, 0.6f, 0.6f, 1.0 };
	GLfloat mat_shininess[] = { 90.0 };
    DisplayLists::createAtomDisplayList();
    qglClearColor(backgroundColor);
    glShadeModel (GL_SMOOTH);
    glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
    glMaterialfv(GL_FRONT, GL_SHININESS, mat_shininess);
    glLightfv(GL_LIGHT0, GL_DIFFUSE, light_diffuse);
    glLightfv(GL_LIGHT0, GL_SPECULAR, light_specular);
    glLightfv(GL_LIGHT0, GL_POSITION, lightPostion);
    glEnable(GL_NORMALIZE);
    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);
    glEnable(GL_DEPTH_TEST);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    
}

void BioStruct3DGLWidget::resizeGL(int width, int height)
{
   glFrame->updateViewPort(width, height);
}

void BioStruct3DGLWidget::paintGL()
{
    if (!isVisible())
        return;
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    draw();
}

void BioStruct3DGLWidget::mousePressEvent(QMouseEvent *event)
{
	lastPos = getTrackballMapping(event->x(), event->y());
}

void BioStruct3DGLWidget::mouseMoveEvent(QMouseEvent *event)
{
    Vector3D rotCenter = biostruc.getCenter();
	
	if (event->buttons() & Qt::LeftButton) {
		Vector3D curPos = getTrackballMapping(event->x(), event->y());
		Vector3D delta = curPos - lastPos;
		if (delta.x || delta.y || delta.z) {
			rotAngle = 90.0f*delta.length();
			rotAxis =  vector_cross(lastPos,curPos); 
            bool syncLock = isSyncModeOn();
            QList<GLFrame*> frames = frameManager->getActiveGLFrameList(glFrame.get(), syncLock);
            foreach( GLFrame* frame, frames) {
                frame->makeCurrent();
                frame->rotateCamera(rotAxis, rotAngle);
                frame->updateGL();
            } 
        }
		lastPos = curPos;
	}
 
}


void BioStruct3DGLWidget::draw()
{
    Vector3D rotCenter = biostruc.getCenter();
    glMatrixMode(GL_MODELVIEW);
	glLoadIdentity(); 
    gluLookAt(0.0, 0.0, cameraDistance, 0,  0, 0, 0.0, 1.0, 0.0);
    glMultMatrixf(glFrame->getRotationMatrix() );
    glTranslatef(-rotCenter.x ,-rotCenter.y, -rotCenter.z);
    clock_t t1 =  clock();
    biostructRenderer->drawBioStruct3D();
    if(NULL != molSurface.get())
    {
        glEnable(GL_BLEND);
        glEnable(GL_BLEND);
        glEnable(GL_CULL_FACE);
        glCullFace(GL_FRONT);
        surfaceRenderer->drawSurface(*molSurface);
        glCullFace(GL_BACK);
        surfaceRenderer->drawSurface(*molSurface);
        glDisable(GL_CULL_FACE);
        glDisable(GL_BLEND);
    }
    clock_t t2 = clock();
    logPerf.trace("BioStruct3D structure rendering time " + QString::number((float)( t2 - t1)/ CLOCKS_PER_SEC)+ " s");

    
}

Vector3D BioStruct3DGLWidget::getTrackballMapping(int x, int y)
{
    Vector3D pos;
	/* project x,y onto a hemisphere centered within width, height */
	pos.x = (2.0f*x - width()) / width();
	pos.y = (height() - 2.0f*y) / height();
	pos.z = 0;
	float d = pos.length();
	d = (d < 1.0) ? d : 1.0;
	pos.z = sqrtf(1.0f - d*d);
	pos.normalize();

	return pos;

} 


void BioStruct3DGLWidget::wheelEvent ( QWheelEvent * event )
{
    float numDegrees =  event->delta() / 8;
    zoom(numDegrees / 10);
}

void BioStruct3DGLWidget::createActions()
{

    QAction* action = NULL;
    
    animationTimer = new QTimer(this);
    animationTimer->setInterval(20); // fixed interval
    connect(animationTimer, SIGNAL(timeout()), this, SLOT(sl_updateAnnimation()));
    
    rendererActions = new QActionGroup(this);
    connect(rendererActions, SIGNAL(triggered(QAction *)), this, SLOT(sl_selectGLRenderer(QAction *)));
    foreach(QString key, rendererFactoryMap.keys()) {
        action  = new QAction(key, rendererActions);
        action->setCheckable(true);
    }

    colorSchemeActions = new QActionGroup(this);	
	connect(colorSchemeActions, SIGNAL(triggered(QAction *)), this, SLOT(sl_selectColorScheme(QAction *)));
    foreach(QString key, colorSchemeFactoryMap.keys()) {
        action = new QAction(key, colorSchemeActions);
        action->setCheckable(true);
    }   
    
    molSurfaceRenderActions = new QActionGroup(this);
    connect(molSurfaceRenderActions, SIGNAL(triggered(QAction *)), this, SLOT(sl_selectSurfaceRenderer(QAction*)));
    foreach(QString key, surfaceRendererFactoryMap.keys()) {
        action = new QAction(key, molSurfaceRenderActions);
        action->setCheckable(true);
        if (key == ConvexMapRenderer::ID) {
            action->setChecked(true);
        }
    }   

    molSurfaceTypeActions = new QActionGroup(this);
    foreach(QString key, AppContext::getMolecularSurfaceFactoryRegistry()->getSurfNameList())
    {
        action = new QAction(key, molSurfaceTypeActions);
        connect(action, SIGNAL(triggered()), this, SLOT(sl_showSurface()));
        action->setCheckable(true);
    }
    action = new QAction(tr("Off"), molSurfaceTypeActions);
    connect(action, SIGNAL(triggered()), this, SLOT(sl_hideSurface()));
    action->setCheckable(true);
    action->setChecked(true);
    
    spinAction = new QAction(tr("Spin"), this);
    spinAction->setCheckable(true);
    connect(spinAction, SIGNAL(triggered()), this, SLOT(sl_acitvateSpin()));
    
    settingsAction = new QAction(tr("Settings"), this);
    connect(settingsAction, SIGNAL(triggered()), this, SLOT(sl_settings()));

    closeAction = new QAction(tr("close_action"), this);
    connect(closeAction, SIGNAL(triggered()), this, SLOT(sl_closeWidget()));

    exportImageAction = new QAction(tr("Export image"), this);
    connect(exportImageAction, SIGNAL(triggered()), this, SLOT(sl_exportImage()));

    if (multipleModels) {
        setNextModelAction = new QAction(tr("Next model"), this);
        connect(setNextModelAction, SIGNAL(triggered()), this, SLOT(sl_setNextModelAcitve()));

        setPrevModelAction = new QAction(tr("Prev model"), this);
        connect(setPrevModelAction, SIGNAL(triggered()), this, SLOT(sl_setPrevModelAcitve()));

        setAllModelsActiveAction = new QAction(tr("All models"), this);
        setAllModelsActiveAction->setCheckable(true);
        connect(setAllModelsActiveAction, SIGNAL(triggered()), this, SLOT(sl_setAllModelsActive()));
        setAllModelsActiveAction->setChecked(true);
    }
}

void BioStruct3DGLWidget::createMenus()
{
    // Renderer selection   
    selectRendererMenu = new QMenu(tr("Render style"));
    selectRendererMenu->addActions(rendererActions->actions());
    
    // Color scheme selection
    selectColorSchemeMenu = new QMenu(tr("Color Scheme"));
	selectColorSchemeMenu->addActions(colorSchemeActions->actions());
    
    // Molecular surface
    QMenu* surfaceMenu = new QMenu(tr("Molecular Surface Render Style"));
    surfaceMenu->addActions(molSurfaceRenderActions->actions());

    QMenu *surfaceTypeMenu = new QMenu(tr("Molecular Surface"));
    surfaceTypeMenu->addActions(molSurfaceTypeActions->actions());
        
    // Models selection 
    if (multipleModels) {
        modelsMenu = new QMenu(tr("Models"));
        modelsMenu->addAction(setNextModelAction);
        modelsMenu->addAction(setPrevModelAction);
        modelsMenu->addAction(setAllModelsActiveAction);
    } else {
        modelsMenu = NULL;
    }

    // Display (context) menu
    displayMenu = new QMenu(this);
    displayMenu->addMenu(selectRendererMenu);
    displayMenu->addMenu(selectColorSchemeMenu);
    if (modelsMenu) {
        displayMenu->addMenu(modelsMenu);
    }
    displayMenu->addMenu(surfaceMenu);
    displayMenu->addMenu(surfaceTypeMenu);
    displayMenu->addAction(spinAction);
    displayMenu->addAction(settingsAction);
    displayMenu->addAction(exportImageAction);

}

void BioStruct3DGLWidget::connectExternalSignals()
{
    AnnotationSettingsRegistry* asr = AppContext::getAnnotationsSettingsRegistry();
    connect(asr, SIGNAL(si_annotationSettingsChanged(const QStringList& )), this, SLOT(sl_updateRenderSettings(const QStringList& )) );

    const QList<ADVSequenceObjectContext*> seqContexts = getSequenceContexts();
    foreach (ADVSequenceObjectContext* ctx, seqContexts) {
        connect(ctx->getSequenceSelection(), 
            SIGNAL(si_selectionChanged(LRegionsSelection*, const QList<LRegion>&, const QList<LRegion>&)), 
            SLOT(sl_onSequenceSelectionChanged(LRegionsSelection*, const QList<LRegion>& , const QList<LRegion>&)));

        connect(dnaView->getAnnotationsSelection(), 
            SIGNAL(si_selectionChanged(AnnotationSelection* , const QList<Annotation*>&, const QList<Annotation*>&)),
            SLOT(sl_onAnnotationSelectionChanged(AnnotationSelection* , const QList<Annotation*>&, const QList<Annotation*>&)));
    }

}



void BioStruct3DGLWidget::setBioStruct3DColorScheme( BioStruct3DColorScheme* clScheme )
{
	Q_ASSERT(clScheme != NULL);
	biostructRenderer->setColorScheme(clScheme);
    colorScheme.reset(clScheme);
   
}

void BioStruct3DGLWidget::setBioStruct3DRenderer( BioStruct3DGLRenderer* r )
{
	Q_ASSERT(r != NULL);
	biostructRenderer.reset(r);

}

void BioStruct3DGLWidget::contextMenuEvent( QContextMenuEvent *event )
{
	QMenu menu;
    
    foreach(QAction* action, getDisplayMenu()->actions()) {
        menu.addAction(action);
    }

    menu.addAction(closeAction);

	menu.exec(event->globalPos());
}

void BioStruct3DGLWidget::sl_selectColorScheme(QAction* action)
{
	BioStruct3DColorScheme* colorScheme = NULL;
	QString schemeName = action->text();
    colorScheme = createCustomColorScheme(schemeName);
	Q_ASSERT(colorScheme != NULL);
        colorScheme->setSelectionColor(selectionColor);
	setBioStruct3DColorScheme(colorScheme);
    currentColorSchemeName = schemeName;
 	updateGL();
		
}

void BioStruct3DGLWidget::sl_updateRenderSettings(const QStringList& list )
{
    Q_UNUSED(list);
    sl_selectColorScheme(colorSchemeActions->checkedAction());
}


const QMap<QString, QColor> BioStruct3DGLWidget::getSecStructAnnotationColors() const
{
    QMap<QString, QColor> colors;
    AnnotationSettingsRegistry* asr = AppContext::getAnnotationsSettingsRegistry();

    foreach (GObject* obj, biostrucDoc->getObjects() ) {
        if (obj->getGObjectType() == GObjectTypes::ANNOTATION_TABLE) {
            AnnotationTableObject* ao = qobject_cast<AnnotationTableObject*>(obj);
            foreach(Annotation* a, ao->getAnnotations()) {
                QString name = a->getAnnotationName();
                if ( name == BioStruct3D::SecStructAnnotationTag) {
                    QString ssName = a->getQualifiers().first().getQualifierValue();
                    AnnotationSettings* as = asr->getAnnotationSettings(ssName);
                    colors.insert(ssName, as->color);
                }
            }
        }
    }

    return colors;
}



const QMap<int, QColor> BioStruct3DGLWidget::getChainColors() const
{
    QMap<int, QColor> colorMap;
    AnnotationSettingsRegistry* asr = AppContext::getAnnotationsSettingsRegistry();
    
	foreach (GObject* obj, biostrucDoc->getObjects() ) {
		if (obj->getGObjectType() == GObjectTypes::ANNOTATION_TABLE) {
			AnnotationTableObject* ao = qobject_cast<AnnotationTableObject*>(obj);
 			foreach(Annotation* a, ao->getAnnotations()) {
 				QString name = a->getAnnotationName();
 				if (name.startsWith(BioStruct3D::MoleculeAnnotationTag)) {
					int chainId = getQualifierValueByName(a, BioStruct3D::ChainIdQualifierName).toInt();
                    Q_ASSERT(chainId != 0);
                    AnnotationSettings* as = asr->getAnnotationSettings(name);
                    colorMap.insert(chainId, as->color);				
 				}
 			}
		}
 	}
	
    return colorMap;
}

void BioStruct3DGLWidget::sl_selectGLRenderer( QAction* action )
{
    QString rendererName = action->text();
    BioStruct3DGLRenderer* renderer = createCustomRenderer(rendererName);
    Q_ASSERT(renderer != NULL);
    setBioStruct3DRenderer(renderer);
    currentGLRendererName = rendererName;
    updateGL();
}

void BioStruct3DGLWidget::setLightPosition( const Vector3D& pos )
{
    lightPostion[0] = pos.x;
    lightPostion[1] = pos.y;
    lightPostion[2] = pos.z;
    lightPostion[3] = 1.0;
}

BioStruct3DColorScheme* BioStruct3DGLWidget::createCustomColorScheme( const QString& name )
{
    if (colorSchemeFactoryMap.contains(name)) {
        return colorSchemeFactoryMap.value(name)->createInstance(this);    
    } else {
        return NULL;
    }

}

BioStruct3DGLRenderer* BioStruct3DGLWidget::createCustomRenderer( const QString& name )
{
    if (rendererFactoryMap.contains(name)) {
        return rendererFactoryMap.value(name)->createInstance(biostruc, colorScheme.get());    
    } else {
        return NULL;
    }
}


const QList<ADVSequenceObjectContext*> BioStruct3DGLWidget::getSequenceContexts() const
{
    return dnaView->getSequenceContexts();
}

void BioStruct3DGLWidget::sl_onSequenceSelectionChanged( LRegionsSelection* s, const QList<LRegion>&  added, const QList<LRegion>&  removed)
{
    if (!isVisible())
        return;

    DNASequenceSelection* selection = qobject_cast<DNASequenceSelection*>(s);
    const DNASequenceObject* seqObj = selection->getSequenceObject();
    
    // TODO: better to use objects relations here
    if (!seqObj->getGObjectName().startsWith(biostruc.pdbId)) {
        return;
    }
    
    
    if (seqObj) {
        int chainId = getSequenceChainId(seqObj);
        Q_ASSERT(biostruc.moleculeMap.contains(chainId));
        colorScheme->updateSelectionRegion(chainId, added, removed);
        biostructRenderer->updateColorScheme();
        update();
    }


}


int BioStruct3DGLWidget::getChainIdForAnnotationObject(AnnotationTableObject* ao) 
{
    if ( this->chainIdCache.contains(ao) ) {
        return chainIdCache.value(ao);
    } else {
        int chainID = -1;
        QList<GObjectRelation> relations = ao->findRelatedObjectsByRole(GObjectRelationRole::SEQUENCE);
        Q_ASSERT(relations.count() == 1);
        GObjectRelation relation = relations.first();
        GObject* obj = biostrucDoc->findGObjectByName(relation.ref.objName);
        DNASequenceObject* seqObj = qobject_cast<DNASequenceObject*>(obj);
        if (seqObj) {
            chainID = getSequenceChainId(seqObj);
            chainIdCache.insert(ao, chainID);
        } 
        return chainID;
    }
}

void BioStruct3DGLWidget::sl_onAnnotationSelectionChanged( AnnotationSelection* as, const QList<Annotation*>& added, const QList<Annotation*>& removed )
{
    Q_UNUSED(as);
    if (!isVisible())
        return;
    
    QList<LRegion> empty;
    QList<GObject*> seqObjects = dnaView->getSequenceGObjectsWithContexts();

	foreach (Annotation* annotation, added) {
        if (annotation->getLocation().isEmpty()) { //TODO: there must be no such annotations!
            continue;
        }

		AnnotationTableObject* ao = annotation->getGObject();
        int chainId  = getChainIdForAnnotationObject(ao);
        if (chainId!=-1) {
		    Q_ASSERT(biostruc.moleculeMap.contains(chainId));
		    colorScheme->updateSelectionRegion(chainId, annotation->getLocation(), empty );
        }
    }
  
    foreach (Annotation* annotation, removed) {
        if (annotation->getLocation().isEmpty() ) { //TODO: there must be no such annotations!
           continue;
        }
	   AnnotationTableObject* ao = annotation->getGObject(); 
	   int chainId  = getChainIdForAnnotationObject(ao);
       if (chainId!=-1) {
           Q_ASSERT(biostruc.moleculeMap.contains(chainId));
           colorScheme->updateSelectionRegion(chainId, empty, annotation->getLocation());
       }
    }
    
    biostructRenderer->updateColorScheme();
    update();
}

QString BioStruct3DGLWidget::getQualifierValueByName( const Annotation* annotation, const QString& qualifierName )
{
    foreach (Qualifier q, annotation->getQualifiers()) {
        if (q.getQualifierName() == qualifierName) {
            return QString(q.getQualifierValue());
        }
    }
    return QString("");
}

void BioStruct3DGLWidget::sl_acitvateSpin()
{
    if (spinAction->isChecked()) { 
        animationTimer->start();
    } else {
        animationTimer->stop();
    }

    updateGL();

}

void BioStruct3DGLWidget::sl_updateAnnimation()
{
    static float velocity = 0.05f;
    spinAngle = velocity* animationTimer->interval();
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    glRotatef(spinAngle, 0, 1, 0 );
    glMultMatrixf(glFrame->getRotationMatrix());
    glGetFloatv( GL_MODELVIEW_MATRIX, glFrame->getRotationMatrix());
    updateGL();
}

#define PRODUCT_NAME "Unipro Ugene"
#define PLUGIN_NAME "BioStruct3D Viewer Plugin"
#define COLOR_SCHEME_NAME "ColorScheme"
#define RENDERER_NAME "GLRenderer"
#define OBJECT_ID_NAME "OBJECT_ID"


QVariantMap BioStruct3DGLWidget::getState()
{
    QVariantMap state;
    glFrame->writeStateToMap(state);
    state[COLOR_SCHEME_NAME] = QVariant::fromValue(currentColorSchemeName);
    state[RENDERER_NAME] = QVariant::fromValue(currentGLRendererName);
    state[OBJECT_ID_NAME] = QVariant::fromValue(getBioStruct3DObjectName());

    return state;

}

void BioStruct3DGLWidget::setState( const QVariantMap& state )
{
    // TODO: draw current selection correctly
    if (state.isEmpty()) {
        return;
    }
    glFrame->setState(state);
    
    currentColorSchemeName = state.value(COLOR_SCHEME_NAME, BioStruct3DColorSchemeFactory::defaultFactoryName()).value<QString>();
    currentGLRendererName = state.value(RENDERER_NAME, BioStruct3DGLRendererFactory::defaultFactoryName()).value<QString>();
    
    setBioStruct3DColorScheme(createCustomColorScheme(currentColorSchemeName));
    setBioStruct3DRenderer(createCustomRenderer(currentGLRendererName));
    resizeGL(width(), height());
    updateGL();

}

const QString BioStruct3DGLWidget::getBioStruct3DObjectName() const
{
    foreach (GObject* obj, biostrucDoc->getObjects()) {
        if (obj->getGObjectType() == GObjectTypes::BIOSTRUCTURE_3D)
            return obj->getGObjectName();
    }
    Q_ASSERT(0);
    return ""; 
}

void BioStruct3DGLWidget::sl_settings()
{
    SettingsDialog dialog;
    dialog.setBackgroundColor(backgroundColor);
    dialog.setSelectionColor(selectionColor);
    if(QDialog::Accepted == dialog.exec())
    {
        backgroundColor=dialog.getBackgroundColor();
        selectionColor=dialog.getSelectionColor();
        colorScheme->setSelectionColor(selectionColor);
        this->makeCurrent();
        qglClearColor(backgroundColor);
        updateGL();
    }
}

void BioStruct3DGLWidget::sl_closeWidget()
{
    hide();
    emit si_widgetClosed(this);
}

void BioStruct3DGLWidget::sl_exportImage() 
{
    
    ExportImageDialog dialog(this);
    dialog.exec();

}

void BioStruct3DGLWidget::zoom( float delta )
{
    bool syncLock = isSyncModeOn();
    QList<GLFrame*> frames = frameManager->getActiveGLFrameList(glFrame.get(), syncLock);
    foreach( GLFrame* frame, frames) {
        frame->makeCurrent();
        frame->performZoom(delta);
        frame->updateViewPort();
        frame->updateGL();
    } 
}

void BioStruct3DGLWidget::saveDefaultSettings()
{
    glFrame->writeStateToMap(defaultsSettings);
    defaultsSettings[COLOR_SCHEME_NAME] = QVariant::fromValue(currentColorSchemeName);
    defaultsSettings[RENDERER_NAME] = QVariant::fromValue(currentGLRendererName);

}

void BioStruct3DGLWidget::restoreDefaultSettigns()
{   
    Q_ASSERT(!defaultsSettings.isEmpty());
    bool syncLock = isSyncModeOn();
    QList<GLFrame*> frames = frameManager->getActiveGLFrameList(glFrame.get(), syncLock);
    foreach( GLFrame* frame, frames) {
        frame->makeCurrent();
        frame->setState(defaultsSettings);
        frame->updateViewPort();
        frame->updateGL();
    } 
}


QMenu* BioStruct3DGLWidget::getDisplayMenu()
{
    Q_ASSERT(displayMenu != NULL);
    return displayMenu;
}



void BioStruct3DGLWidget::createActiveModelIndexList()
{
    int numModels = biostruc.modelMap.count();
    if (numModels > 1) {
        multipleModels = true;
        // set all models active
        for (int i = 0; i < numModels; ++i) {
            activeModelIndexList.append(i);
        }
    } else {
        multipleModels = false;
        activeModelIndexList.append(0);
    }
    
}


void BioStruct3DGLWidget::switchActiveModel(bool forward)
{
    if (activeModelIndexList.count() == 1) {
        int index = activeModelIndexList.takeFirst();
        int numModels = biostruc.modelMap.size();
        if (forward) {
            ++index;
        } else {
            --index;
            if (index == -1 ) {
                index =numModels - 1;
            }
        }
        activeModelIndexList.append(index);
    } else {
        // if there are more then one model active, unselect all, set default index = 0
        activeModelIndexList.clear();
        activeModelIndexList.append(0);
        setAllModelsActiveAction->setChecked(false);
    }
}


void BioStruct3DGLWidget::sl_setNextModelAcitve()
{
    switchActiveModel(true);
    updateGL();
}


void BioStruct3DGLWidget::sl_setPrevModelAcitve()
{
    switchActiveModel(false);
    updateGL();
}

void BioStruct3DGLWidget::sl_setAllModelsActive()
{
    int numModels = biostruc.modelMap.count();
    activeModelIndexList.clear();
    for (int i = 0; i < numModels; ++i) {
        activeModelIndexList.append(i);
    }
    updateGL();

}


void BioStruct3DGLWidget::writeImage2DToFile( int format, int options, int nbcol, const char *fileName )
{
    FILE *fp = NULL;
    const char* FOPEN_ARGS = "wb";
    const QByteArray title(fileName);
    int state = GL2PS_OVERFLOW, buffsize = 0;
    GLint viewport[4];
    int sort = GL2PS_SIMPLE_SORT;
        
    fp = fopen(fileName, FOPEN_ARGS);

    if(!fp){
        QMessageBox::warning(this, tr("Error"),tr("Unable to open file %1 for writing").arg(fileName));
        return;
    }
    
    glGetIntegerv(GL_VIEWPORT,viewport);
    
    if (format == GL2PS_EPS) {
        // hack -> make widget aspect ratio 1:1
        if (width() > height()) {
            int size = height();
            resize(size, size);
        }
    }

    while(state == GL2PS_OVERFLOW){
        buffsize += 2048*2048;
        gl2psBeginPage(title.constData(), "Unipro UGENE BioStruct3D Viewer plugin", viewport, format, sort, options,
            GL_RGBA, 0, NULL, nbcol, nbcol, nbcol, buffsize, fp, fileName);
        paintGL();
        state = gl2psEndPage();
    }

    fclose(fp);
    
    if (format == GL2PS_EPS) {
        // restore sizes 
        updateGeometry();
    }
  
}

const QString BioStruct3DGLWidget::getPDBId() const
{
    return biostruc.pdbId;
}

int BioStruct3DGLWidget::getSequenceChainId( const DNASequenceObject* seqObj )
{
    const QVariantMap info =  seqObj->getDNASequence().info;
    if (info.contains(DNAInfo::CHAIN_ID)) {
        return info.value(DNAInfo::CHAIN_ID).toInt();
    }

    return -1;
}

void BioStruct3DGLWidget::loadColorSchemes()
{
    currentColorSchemeName =  BioStruct3DColorSchemeFactory::defaultFactoryName();

    QList<QAction*>::iterator iter;
    QList<QAction*> schemeActions = colorSchemeActions->actions();
    for (iter = schemeActions.begin(); iter != schemeActions.end(); ++iter) {
        if ((*iter)->text() == currentColorSchemeName) {
            (*iter)->setChecked(true);
            break;
        }
    }
    Q_ASSERT(iter != schemeActions.end());
    colorScheme.reset(colorSchemeFactoryMap.value(currentColorSchemeName)->createInstance(this));
    colorScheme->setSelectionColor(selectionColor);
}

void BioStruct3DGLWidget::loadGLRenderers()
{
    currentGLRendererName =  BioStruct3DGLRendererFactory::defaultFactoryName();
    QList<QAction*>::iterator iter;
    QList<QAction*> renderActions = rendererActions->actions();
    for (iter = renderActions.begin(); iter != renderActions.end(); ++iter) {
        if ((*iter)->text() == currentGLRendererName) {
            (*iter)->setChecked(true);
            break;
        }
    }
    Q_ASSERT(iter != renderActions.end());
    const BioStruct3DColorScheme* scheme = colorScheme.get();  
    biostructRenderer.reset(rendererFactoryMap.value(currentGLRendererName)->createInstance(biostruc,scheme));
    
    QString surfaceRendererName = ConvexMapRenderer::ID;
    surfaceRenderer.reset(surfaceRendererFactoryMap.value(surfaceRendererName)->createInstance());
    
}

bool BioStruct3DGLWidget::isSyncModeOn()
{
    Qt::KeyboardModifiers km = QApplication::keyboardModifiers();
    bool synchronizationMode = km.testFlag(Qt::ShiftModifier) || frameManager->getSyncLock();
    synchronizationMode &=  frameManager->getGLFrames().count() > 1 ;
    return synchronizationMode;
}

void BioStruct3DGLWidget::sl_showSurface()
{
    {
        // Calculate surface (only one model! too much computation for all models)
        {
            MolecularSurfaceFactory *factory=
                    AppContext::getMolecularSurfaceFactoryRegistry()->getSurfaceFactory(qobject_cast<QAction *>(sender())->text());
            molSurface.reset(factory->createInstance());
        }
        QList<SharedAtom> atoms;
        int index = getActiveModelIndexList().first();
        foreach (const SharedMolecule mol, biostruc.moleculeMap) {
            const Molecule3DModel& model = mol->models.at(index);
            atoms += model.atoms;
        }
        molSurface->calculate(atoms);
    }
    
    makeCurrent();
    updateGL();
}

void BioStruct3DGLWidget::sl_hideSurface()
{
    molSurface.reset();

    makeCurrent();
    updateGL();
}

void BioStruct3DGLWidget::sl_selectSurfaceRenderer(QAction*  action)
{
    {
        MolecularSurfaceRendererFactory* f = surfaceRendererFactoryMap.value(action->text());
        surfaceRenderer.reset( f->createInstance() );
    }

    makeCurrent();
    updateGL();
}

} //namespace
