/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtCore/QtCore>
#include <QtCore/QThread>
#include <QtCore/QDebug>

#include <core_api/Log.h>
#include <core_api/AppContext.h>
#include <core_api/DNATranslation.h>
#include <core_api/DocumentModel.h>
#include <core_api/Counter.h>
#include <util_text/TextUtils.h>
#include <util_tasks/CreateAnnotationTask.h>
#include <datatype/AnnotationData.h>

#include "ScriptEngineContext.h"
#include "RemoteRequestTask.h"

namespace GB2
{
/* TRANSLATOR GB2::RemoteRequestTask */ 

static LogCategory log(ULOG_CAT_PLUGIN_REMOTE_REQUEST);

RemoteRequestToAnnotationsTask::RemoteRequestToAnnotationsTask( Script * _script, int _maxrl, int _minrl, SendSelectionStrand _strand, 
                                     DNATranslation * _complT, DNATranslation * _aminoT,  const QByteArray & _query, 
                                     int _qoffs, AnnotationTableObject* _ao, const QString & _group  ) :
Task( tr("remote_request_task"), TaskFlags_NR_FOSCOE ), offsInGlobalSeq(_qoffs), aobj(_ao), group(_group)
{
    GCOUNTER( cvar, tvar, "RemoteRequestToAnnotationsTask" );
    RemoteRequestTaskSettings cfg;
    cfg.script = _script;
    cfg.maxrl = _maxrl;
    cfg.minrl = _minrl;
    cfg.strand = _strand;
    cfg.complT = _complT;
    cfg.aminoT = _aminoT;
    cfg.query = _query;

    queryTask = new RemoteRequestTask(cfg);
    addSubTask(queryTask);
}

QList<Task*> RemoteRequestToAnnotationsTask::onSubTaskFinished(Task* subTask) {
    QList<Task*> res;
    if (hasErrors() || isCanceled()) {
        return res;
    }

    if (aobj.isNull()) {
        stateInfo.setError(  tr("obj_was_removed\n") );
        return res;
    }
    if (subTask == queryTask) {
        //shift annotations according to offset first
        RemoteRequestTask * rrTask = qobject_cast<RemoteRequestTask *>(queryTask);
        assert( rrTask );

        QList<SharedAnnotationData> anns = rrTask->getResultedAnnotations();

        for(QMutableListIterator<SharedAnnotationData> it_ad(anns); it_ad.hasNext(); ) {
            AnnotationData * ad = it_ad.next().data();
            QList<LRegion> & regions = ad->location;
            for( QMutableListIterator<LRegion> it(regions); it.hasNext(); ) {
                it.next().startPos += offsInGlobalSeq;
            }
        }

        res.append(new CreateAnnotationsTask(aobj, group, anns));
    }
    return res;
}

RemoteRequestTask::RemoteRequestTask( const RemoteRequestTaskSettings & cfg_) :
Task( tr("remote_request_task"), TaskFlag_None ), cfg(cfg_)
{
    engine = new QScriptEngine();
}


void RemoteRequestTask::prepareQueries() {
    bool dir = (cfg.strand == SendSelectionStrand_Both || cfg.strand == SendSelectionStrand_Direct);
    bool complement = (cfg.strand == SendSelectionStrand_Both || cfg.strand == SendSelectionStrand_Complement) && cfg.complT;
    if( complement ) {
        Query q;
        q.complement = true;

        QByteArray complQuery( cfg.query.size(), 0 );
        cfg.complT->translate( cfg.query.data(), cfg.query.size(), complQuery.data(), complQuery.size() );
        TextUtils::reverse( complQuery.data(), complQuery.size() );
        if( cfg.aminoT ) {
            q.amino = true;
            for( int i = 0; i < 3; ++i ) {
                QByteArray aminoQuery( cfg.query.size() / 3, 0 );
                cfg.aminoT->translate( complQuery.data() + i, complQuery.size()-i, aminoQuery.data(), aminoQuery.size() );
                q.seq = aminoQuery;
                q.offs = i;
                queries.push_back(q);
            }            
        } else {
            q.seq = complQuery;
            queries.push_back(q);
        }

    } 
    if( dir ) {
        Query q;
        if( cfg.aminoT ) {
            q.amino = true;
            for( int i = 0; i < 3; ++i ) {
                QByteArray aminoQuery( cfg.query.size() / 3, 0 );
                cfg.aminoT->translate( cfg.query.data()+i, cfg.query.size()-i, aminoQuery.data(), aminoQuery.size() );
                q.seq = aminoQuery;
                q.offs = i;
                queries.push_back(q);
            }
        } else {
            q.seq = cfg.query;
            queries.push_back(q);
        }
    } 
}

void RemoteRequestTask::prepareEngine() {
    cfg.script->init_engine( engine );
    ScriptHttpAnnotatorContext::setDefaultProperties( engine );
    ScriptHttpAnnotatorContext::setMaxResLen( engine, cfg.maxrl );
    ScriptHttpAnnotatorContext::setMinResLen( engine, cfg.minrl );
    ScriptHttpAnnotatorContext::setLog( engine, &log );
    ScriptHttpAnnotatorContext::setTaskStateInfo( engine, &stateInfo );
    cfg.script->callSetup(engine);
}

void RemoteRequestTask::run() {
    prepareQueries();

    foreach( Query q, queries) {
        if( isCanceled() ) {
            return;
        }
        prepareEngine();
        ScriptHttpAnnotatorContext::setQuery( engine, QString(q.seq) );
        QScriptValue exception = engine->nullValue();
        cfg.script->callMain( engine, &exception );
        if( !exception.isNull() ) {
            QString exc_msg = exception.isError() ? exception.property("message").toString() : exception.toString();
            log.error( tr("script_reports_error: ") + exc_msg );
            QStringList backtrace = engine->uncaughtExceptionBacktrace();
            if( !backtrace.isEmpty() ) {
                log.details( tr("exception_backtrace:") );
                foreach( QString s, backtrace ) {
                    log.details(s);
                }
            }
            break;
        }
        createAnnotations(q);
    }
}

void RemoteRequestTask::cleanup() {
    delete engine;
}

QList<SharedAnnotationData> RemoteRequestTask::getResultedAnnotations() const {
    return resultAnnotations;
}

void RemoteRequestTask::createAnnotations( const Query & q ) {
    QList<SharedAnnotationData> annotations = ScriptHttpAnnotatorContext::getAnnotations( engine );
    if ( annotations.isEmpty() ) {
        return;
    }
    for( QList<SharedAnnotationData>::iterator it = annotations.begin(), end = annotations.end(); end != it; ++it ) {
        for( QList<LRegion>::iterator jt = it->data()->location.begin(), eend = it->data()->location.end(); 
            eend != jt; ++jt ) {
                int & s = jt->startPos;
                int & l = jt->len;

                if( q.complement ) {
                    s = q.seq.size() - s - l;
                    it->data()->complement = !it->data()->complement;
                }
                if( q.amino ) {
                    s = s * 3 + (q.complement ? 2 - q.offs : q.offs);
                    l = l * 3;
                }
        }
    }
    resultAnnotations << annotations;
}
} //namespace
