/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/


#include <memory>

#include <QtCore/QList>

#include <core_api/DocumentModel.h>
#include <core_api/IOAdapter.h>
#include <core_api/AppContext.h>
#include <gobjects/DNASequenceObject.h>
#include <util_text/TextUtils.h>

#include "uhmmer3SearchTests.h"

namespace GB2 {

/*******************************
* GTest_GeneralUHMM3Search
********************************/

const QString GTest_UHMM3Search::SEQ_DOC_CTX_NAME_TAG        = "seqDoc";
const QString GTest_UHMM3Search::HMM_FILENAME_TAG            = "hmm";
const QString GTest_UHMM3Search::HMMSEARCH_TASK_CTX_NAME_TAG = "taskCtxName";
const QString GTest_UHMM3Search::ALGORITHM_TYPE_OPTION_TAG   = "algo";

const QString GTest_UHMM3Search::SEQ_E_OPTION_TAG            = "seqE";
const QString GTest_UHMM3Search::SEQ_T_OPTION_TAG            = "seqT";
const QString GTest_UHMM3Search::Z_OPTION_TAG                = "z";
const QString GTest_UHMM3Search::DOM_E_OPTION_TAG            = "domE";
const QString GTest_UHMM3Search::DOM_T_OPTION_TAG            = "domT";
const QString GTest_UHMM3Search::DOM_Z_OPTION_TAG            = "domZ";
const QString GTest_UHMM3Search::USE_BIT_CUTOFFS_OPTION_TAG  = "ubc";
const QString GTest_UHMM3Search::INC_SEQ_E_OPTION_TAG        = "incE";
const QString GTest_UHMM3Search::INC_SEQ_T_OPTION_TAG        = "incT";
const QString GTest_UHMM3Search::INC_DOM_E_OPTION_TAG        = "incdomE";
const QString GTest_UHMM3Search::INC_DOM_T_OPTION_TAG        = "incdomT";
const QString GTest_UHMM3Search::INC_USE_BIT_CUTOFFS_OPTION_TAG = "incubc";
const QString GTest_UHMM3Search::MAX_OPTION_TAG              = "max";
const QString GTest_UHMM3Search::F1_OPTION_TAG               = "f1";
const QString GTest_UHMM3Search::F2_OPTION_TAG               = "f2";
const QString GTest_UHMM3Search::F3_OPTION_TAG               = "f3";
const QString GTest_UHMM3Search::NOBIAS_OPTION_TAG           = "nobias";
const QString GTest_UHMM3Search::NONULL2_OPTION_TAG          = "nonull2";
const QString GTest_UHMM3Search::SEED_OPTION_TAG             = "seed";

static void setDoubleOption( double& num, const QDomElement& el, const QString& optionName, TaskStateInfo& si ) {
    if( si.hasErrors() ) {
        return;
    }
    QString numStr = el.attribute( optionName );
    if( numStr.isEmpty() ) {
        return;
    }
    
    bool ok = false;
    double ret = numStr.toDouble( &ok );
    if( !ok ) {
        si.setError( QString( "cannot_parse_double_number_from %1. Option: %2" ).arg( numStr ).arg( optionName ) );
        return;
    }
    num = ret;
}

static void setUseBitCutoffsOption( int& ret, const QDomElement& el, const QString& opName, TaskStateInfo& si ) {
    if( si.hasErrors() ) {
        return;
    }

    QString str = el.attribute( opName ).toLower();

    if( "ga" == str ) {
        ret = p7H_GA;
    } else if( "nc" == str ) {
        ret = p7H_NC;
    } else if( "tc" == str ) {
        ret = p7H_TC;
    } else if( !str.isEmpty() ) {
        si.setError( QString( "unrecognized_value_in %1 option" ).arg( opName ) );
    }
}

static void setBooleanOption( int& ret, const QDomElement& el, const QString& opName, TaskStateInfo& si ) {
    if( si.hasErrors() ) {
        return;
    }
    QString str = el.attribute( opName ).toLower();

    if( !str.isEmpty() &&  "n" != str && "no" != str ) {
        ret = TRUE;
    } else {
        ret = FALSE;
    }
}

static void setIntegerOption( int& num, const QDomElement& el, const QString& optionName, TaskStateInfo& si ) {
    if( si.hasErrors() ) {
        return;
    }
    QString numStr = el.attribute( optionName );
    if( numStr.isEmpty() ) {
        return;
    }

    bool ok = false;
    int ret = numStr.toInt( &ok );
    if( !ok ) {
        si.setError( QString( "cannot_parse_integer_number_from %1. Option: %2" ).arg( numStr ).arg( optionName ) );
        return;
    }
    num = ret;
}

void GTest_UHMM3Search::setSearchTaskSettings( UHMM3SearchSettings& settings, const QDomElement& el, TaskStateInfo& si ) {
    setDoubleOption( settings.e,       el, GTest_UHMM3Search::SEQ_E_OPTION_TAG,     si );
    setDoubleOption( settings.t,       el, GTest_UHMM3Search::SEQ_T_OPTION_TAG,     si );
    setDoubleOption( settings.z,       el, GTest_UHMM3Search::Z_OPTION_TAG,         si );
    setDoubleOption( settings.f1,      el, GTest_UHMM3Search::F1_OPTION_TAG,        si );
    setDoubleOption( settings.f2,      el, GTest_UHMM3Search::F2_OPTION_TAG,        si );
    setDoubleOption( settings.f3,      el, GTest_UHMM3Search::F3_OPTION_TAG,        si );
    setDoubleOption( settings.domE,    el, GTest_UHMM3Search::DOM_E_OPTION_TAG,     si );
    setDoubleOption( settings.domT,    el, GTest_UHMM3Search::DOM_T_OPTION_TAG,     si );
    setDoubleOption( settings.domZ,    el, GTest_UHMM3Search::DOM_Z_OPTION_TAG,     si );
    setDoubleOption( settings.incE,    el, GTest_UHMM3Search::INC_SEQ_E_OPTION_TAG, si );
    setDoubleOption( settings.incT,    el, GTest_UHMM3Search::INC_SEQ_T_OPTION_TAG, si );
    setDoubleOption( settings.incDomE, el, GTest_UHMM3Search::INC_DOM_E_OPTION_TAG, si );
    setDoubleOption( settings.incDomT, el, GTest_UHMM3Search::INC_DOM_T_OPTION_TAG, si );

    setBooleanOption( settings.doMax,        el, GTest_UHMM3Search::MAX_OPTION_TAG,     si );
    setBooleanOption( settings.noBiasFilter, el, GTest_UHMM3Search::NOBIAS_OPTION_TAG,  si );
    setBooleanOption( settings.noNull2,      el, GTest_UHMM3Search::NONULL2_OPTION_TAG, si );

    setIntegerOption( settings.seed, el, GTest_UHMM3Search::SEED_OPTION_TAG, si );

    setUseBitCutoffsOption( settings.useBitCutoffs, el, GTest_UHMM3Search::USE_BIT_CUTOFFS_OPTION_TAG, si );
    setUseBitCutoffsOption( settings.incuseBitCutoffs, el, GTest_UHMM3Search::INC_USE_BIT_CUTOFFS_OPTION_TAG, si );
}

static void setSearchAlgoType( GTest_UHMM3SearchAlgoType& alType, const QString& s ) {
    QString str = s.toLower();

    if( "general" == str ) {
        alType = GENERAL_SEARCH;
    } else if( "sw" == str ) {
        alType = SEQUENCE_WALKER_SEARCH;
    } else {
        alType = UNKNOWN_SEARCH;
    }
}

void GTest_UHMM3Search::init( XMLTestFormat *tf, const QDomElement& el ) {
    Q_UNUSED( tf );

    hmmFilename = el.attribute( HMM_FILENAME_TAG );
    searchTaskCtxName = el.attribute( HMMSEARCH_TASK_CTX_NAME_TAG );

    generalSearchTask   = NULL;
    swSearchTask        = NULL;
    searchTaskToCtx     = NULL;

    seqDocCtxName = el.attribute( SEQ_DOC_CTX_NAME_TAG );
    setSearchAlgoType( algo, el.attribute( ALGORITHM_TYPE_OPTION_TAG ) );
    setSearchTaskSettings( settings.inner, el, stateInfo );
    
    cleanuped = false;
}

void GTest_UHMM3Search::setAndCheckArgs() {
    assert( !stateInfo.hasErrors() );
    if( hmmFilename.isEmpty() ) {
        stateInfo.setError( "hmm_filename_is_empty" );
        return;
    }
    hmmFilename = env->getVar( "COMMON_DATA_DIR" ) + "/" + hmmFilename;

    if( searchTaskCtxName.isEmpty() ) {
        stateInfo.setError( "task_ctx_name_is_empty" );
        return;
    }

    if( seqDocCtxName.isEmpty() ) {
        stateInfo.setError( "sequence_document_ctx_name_is_empty" );
        return;
    }

    if( UNKNOWN_SEARCH == algo ) {
        stateInfo.setError( "unknown_algorithm_type" );
        return;
    }

    Document* seqDoc = getContext<Document>( this, seqDocCtxName );
    if( NULL == seqDoc ) {
        stateInfo.setError( QString( "context %1 not found" ).arg( seqDocCtxName ) );
        return;
    }
    QList< GObject* > objsList = seqDoc->findGObjectByType( GObjectTypes::DNA_SEQUENCE );
    if( objsList.isEmpty() ) {
        stateInfo.setError( "no_dna_sequence_objects_in_document" );
        return;
    }
    DNASequenceObject* seqObj = qobject_cast< DNASequenceObject* >( objsList.first() );
    if( NULL == seqObj ) {
        stateInfo.setError( "cannot_cast_to_dna_object" );
        return;
    }
    sequence = seqObj->getDNASequence();
    if( !sequence.length() ) {
        stateInfo.setError( "empty_sequence_given" );
        return;
    }
}

void GTest_UHMM3Search::prepare() {
    assert( !hasErrors() );
    setAndCheckArgs();
    if( hasErrors() ) {
        return;
    }

    switch( algo ) {
    case GENERAL_SEARCH:
        generalSearchTask = new UHMM3SearchTask( settings, hmmFilename, sequence.seq.data(), sequence.seq.size() );
        searchTaskToCtx = generalSearchTask;
        addSubTask( generalSearchTask );
        break;
    case SEQUENCE_WALKER_SEARCH:
        swSearchTask = new UHMM3SWSearchTask( hmmFilename, sequence, settings );
        searchTaskToCtx = swSearchTask;
        addSubTask( swSearchTask );
        break;
    default:
        assert( 0 && "undefined_algorithm_type" );
    }
}

Task::ReportResult GTest_UHMM3Search::report() {
    if( stateInfo.hasErrors() ) {
        return ReportResult_Finished;
    }
    assert( NULL != searchTaskToCtx );
    if( !searchTaskToCtx->hasErrors() && !searchTaskToCtx->isCanceled() ) {
        addContext( searchTaskCtxName, searchTaskToCtx );
    }
    return ReportResult_Finished;
}

void GTest_UHMM3Search::cleanup() {
    if( NULL != searchTaskToCtx && !searchTaskToCtx->hasErrors() && !cleanuped ) {
        removeContext( searchTaskCtxName );
        cleanuped = true;
    }
}

GTest_UHMM3Search::~GTest_UHMM3Search() {
}

/**************************
* GTest_GeneralUHMM3SearchCompare
**************************/

const int   BUF_SZ      = 2048;
const char  TERM_SYM    = '\0';

static void readLine( IOAdapter* io, QByteArray& to, QStringList* tokens = NULL ) {
    assert( NULL != io );
    to.clear();
    QByteArray buf( BUF_SZ, TERM_SYM );
    bool there = false;
    int bytes = 0;
    while( !there ) {
        int ret = io->readUntil( buf.data(), BUF_SZ, TextUtils::LINE_BREAKS, IOAdapter::Term_Include, &there );
        if( 0 > ret ) {
            throw QString( "read_error_occurred" );
        }
        if( 0 == ret ) {
            break;
        }
        to.append( QByteArray( buf.data(), ret ) );
        bytes += ret;
    }
    to = to.trimmed();
    if( 0 == bytes ) {
        throw QString( "unexpected_end_of_file_found" );
    }

    if( NULL != tokens ) {
        *tokens = QString( to ).split( QRegExp( "\\s+" ), QString::SkipEmptyParts );
    }
}

static QByteArray getNextToken( QStringList& tokens ) {
    if( tokens.isEmpty() ) {
        throw QString( "unexpected_end_of_line:token_is_missing" );
    }
    return tokens.takeFirst().toAscii();
}

static double getDouble( const QByteArray& numStr ) {
    bool ok = false;
    double ret = numStr.toDouble( &ok );
    if( ok ) {
        return ret;
    }
    throw QString( GTest_UHMM3SearchCompare::tr( "cannot_parse_double_number_from_string:%1" ).arg( QString( numStr ) ) );
}

static float getFloat( const QByteArray& numStr ) {
    return (float)getDouble( numStr );
}

static bool getSignificance( const QByteArray& str ) {
    if( "!" == str ) {
        return true;
    } else if( "?" == str ) {
        return false;
    }
    throw QString( GTest_UHMM3SearchCompare::tr( "cannot_parse_significance:%1" ).arg( QString( str ) ) );
}

static UHMM3SearchSeqDomainResult getDomainRes( QStringList& tokens ) {
    UHMM3SearchSeqDomainResult res;

    getNextToken( tokens );
    res.isSignificant = getSignificance( getNextToken( tokens ) );
    res.score   = getFloat( getNextToken( tokens ) );
    res.bias    = getFloat( getNextToken( tokens ) );
    res.cval    = getDouble( getNextToken( tokens ) );
    res.ival    = getDouble( getNextToken( tokens ) );

    int hmmFrom = (int)getFloat( getNextToken( tokens ) );
    int hmmTo   = (int)getFloat( getNextToken( tokens ) );
    res.queryRegion = LRegion( hmmFrom, hmmTo - hmmFrom + 1 );
    getNextToken( tokens );

    int aliFrom = (int)getFloat( getNextToken( tokens ) );
    int aliTo   = (int)getFloat( getNextToken( tokens ) );
    res.seqRegion = LRegion( aliFrom, aliTo - aliFrom + 1 );
    getNextToken( tokens );

    int envFrom = (int)getFloat( getNextToken( tokens ) );
    int envTo   = (int)getFloat( getNextToken( tokens ) );
    res.envRegion = LRegion( envFrom, envTo - envFrom + 1 );
    getNextToken( tokens );

    res.acc = getDouble( getNextToken( tokens ) );
    return res;
}

static int getExp( double d ) {
    QString str = QString().sprintf( "%e", d ).toLower();
    QStringList strList = str.split( "e", QString::SkipEmptyParts );
    assert( 2 == strList.size() );
    return strList.at( 1 ).toInt();
}

static bool compareFloats( float f1, float f2 ) {
    return qAbs( f1 - f2 ) < 0.1;
}

const double DOUBLE_EPS = 1.0e-20;

static bool compareDoubles( double f1, double f2 ) {
    f1 += DOUBLE_EPS;
    f2 += DOUBLE_EPS;
    int exp1 = getExp( qAbs( f1 ) );
    int exp2 = getExp( qAbs( f2 ) );
    
    if( exp1 < 2 ) {
        return compareFloats( f1, f2 );
    }
    
    if( exp1 != exp2 ) {
        return false;
    }
    f1 = ( 1 <= exp1 )? f1 * qPow( 10, exp1 - 1 ) : f1;
    f2 = ( 1 <= exp2 )? f2 * qPow( 10, exp2 - 1 ) : f2;
    return qAbs( f1 - f2 ) < 0.1;
}

void GTest_UHMM3SearchCompare::generalCompareResults( const UHMM3SearchResult& myRes, const UHMM3SearchResult& trueRes, TaskStateInfo& ti ) {
    const UHMM3SearchCompleteSeqResult& myFull = myRes.fullSeqResult;
    const UHMM3SearchCompleteSeqResult& trueFull = trueRes.fullSeqResult;

    if( myFull.isReported != trueFull.isReported ) {
        ti.setError( "reported_flag_not_matched" );
        return;
    }
    if( myFull.isReported ) {
        if( !compareFloats( myFull.bias, trueFull.bias )   ) { ti.setError( "full_seq_bias_not_matched" );  return; }
        if( !compareDoubles( myFull.eval, trueFull.eval )  ) { ti.setError( "full_seq_eval_not_matched" );  return; }
        if( !compareFloats( myFull.score, trueFull.score ) ) { ti.setError( "full_seq_score_not_matched" ); return; }
        if( !compareFloats( myFull.expectedDomainsNum, trueFull.expectedDomainsNum ) ) { ti.setError( "full_seq_exp_not_matched" ); return; }
        if( myFull.reportedDomainsNum != trueFull.reportedDomainsNum ) { ti.setError( "full_seq_N_not_matched"   ); return; }
    }

    const QList< UHMM3SearchSeqDomainResult >& myDoms = myRes.domainResList;
    const QList< UHMM3SearchSeqDomainResult >& trueDoms = trueRes.domainResList;
    if( myDoms.size() != trueDoms.size() ) {
        ti.setError( "domain_res_number_not_matched" );
        return;
    }
    for( int i = 0; i < myDoms.size(); ++i ) {
        UHMM3SearchSeqDomainResult myCurDom = myDoms.at( i );
        UHMM3SearchSeqDomainResult trueCurDom = trueDoms.at( i );
        if( !compareDoubles( myCurDom.acc, trueCurDom.acc ) )   { ti.setError( "dom_acc_not_matched" );   return; }
        if( !compareFloats( myCurDom.bias, trueCurDom.bias ) )  { ti.setError( "dom_bias_not_matched" );  return; }
        if( !compareDoubles( myCurDom.cval, trueCurDom.cval ) )  { ti.setError( "dom_cval_not_matched at %1" );  return; }
        if( !compareDoubles( myCurDom.ival, trueCurDom.ival ) )  { ti.setError( "dom_ival_not_matched" );  return; }
        if( !compareFloats( myCurDom.score, trueCurDom.score ) ) { ti.setError( "dom_score_not_matched" ); return; }
        if( myCurDom.envRegion != trueCurDom.envRegion ) { ti.setError( "dom_env_region_not_matched" ); return; }
        if( myCurDom.queryRegion != trueCurDom.queryRegion ) { ti.setError( "dom_hmm_region_not_matched" ); return; }
        if( myCurDom.seqRegion != trueCurDom.seqRegion ) { ti.setError( "dom_seq_region_not_matched" ); return; }
        if( myCurDom.isSignificant != trueCurDom.isSignificant ) { ti.setError( "dom_sign_not_matched" ); return; }
    }
}

/* we don't compare seq and env regions here - it doesn't make sence */
static bool findDomainInSWResult( const UHMM3SWSearchTaskResult& res, const UHMM3SearchSeqDomainResult& dRes ) {
    foreach( const UHMM3SWSearchTaskDomainResult& domainRes, res ) {
        UHMM3SearchSeqDomainResult dom = domainRes.generalResult;
        if( !compareDoubles( dom.acc, dRes.acc ) )    { continue; }
        if( !compareFloats( dom.bias, dRes.bias ) )   { continue; }
        if( !compareDoubles( dom.cval, dRes.cval ) )  { continue; }
        if( !compareDoubles( dom.ival, dRes.ival ) )  { continue; }
        if( !compareFloats( dom.score, dRes.score ) ) { continue; }
        if( dom.queryRegion != dRes.queryRegion ) { continue; }
        if( dom.isSignificant != dRes.isSignificant ) { continue; }
        return true;
    }
    return false;
}

/* we compare here that every domain of trueResult is included in myResult */
void GTest_UHMM3SearchCompare::swCompareResults( const UHMM3SWSearchTaskResult& myR, const UHMM3SearchResult& trueR, TaskStateInfo& ti ) {
    int sz = trueR.domainResList.size();
    int i = 0;
    for( i = 0; i < sz; ++i ) {
        bool ok = findDomainInSWResult( myR, trueR.domainResList.at( i ) );
        if( !ok ) {
            ti.setError( QString( "cannot_find_#%1_result" ).arg( i ) );
            return;
        }
    }
}

const QString GTest_UHMM3SearchCompare::SEARCH_TASK_CTX_NAME_TAG = "searchTask";
const QString GTest_UHMM3SearchCompare::TRUE_OUT_FILE_TAG        = "trueOut";

void GTest_UHMM3SearchCompare::init( XMLTestFormat *tf, const QDomElement& el ) {
    Q_UNUSED( tf );

    searchTaskCtxName = el.attribute( SEARCH_TASK_CTX_NAME_TAG );
    trueOutFilename = el.attribute( TRUE_OUT_FILE_TAG );
}

void GTest_UHMM3SearchCompare::setAndCheckArgs() {
    assert( !hasErrors() );

    if( searchTaskCtxName.isEmpty() ) {
        stateInfo.setError( "search_task_ctx_name_is_empty" );
        return;
    }

    if( trueOutFilename.isEmpty() ) {
        stateInfo.setError( "true_out_filename_is_empty" );
        return;
    }
    trueOutFilename = env->getVar( "COMMON_DATA_DIR" ) + "/" + trueOutFilename;

    Task* searchTask = getContext<Task>( this, searchTaskCtxName );
    if( NULL == searchTask ) {
        stateInfo.setError( tr( "cannot_find_search_task_in_context" ) );
        return;
    }

    generalTask = qobject_cast< UHMM3SearchTask*    >( searchTask );
    swTask      = qobject_cast< UHMM3SWSearchTask*  >( searchTask );

    if( NULL != generalTask ) {
        assert( NULL == swTask );
        algo = GENERAL_SEARCH;
    } else if( NULL != swTask ) {
        assert( NULL == generalTask );
        algo = SEQUENCE_WALKER_SEARCH;
    } else {
        assert( 0 && "cannot_cast_task_to_search_task" );
    }
}

UHMM3SearchResult GTest_UHMM3SearchCompare::getOriginalSearchResult( const QString & filename ) {
    assert( !filename.isEmpty() );
    
    IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById( BaseIOAdapters::url2io( filename ) );
    std::auto_ptr< IOAdapter > io( iof->createIOAdapter() );
    if( NULL == io.get() ) {
        throw QString( "cannot_create_io_adapter_for_'%1'_file" ).arg( filename );
    }
    if( !io->open( filename, IOAdapterMode_Read ) ) {
        throw QString( "cannot_open_'%1'_file" ).arg( filename );
    }
    
    UHMM3SearchResult res;
    QByteArray buf;
    QStringList tokens;
    bool wasHeader = false;
    bool wasFullSeqResult = false;
    readLine( io.get(), buf ); /* the first line. starts with # search or # phmmer */
    do {
        readLine( io.get(), buf );
        if( buf.isEmpty() ) { /* but no error - empty line here */
            continue;
        }
        if( buf.startsWith( "# HMMER 3" ) ) {
            wasHeader = true;
            continue;
        }
        if( buf.startsWith( "Scores for complete sequences" ) ) {
            if( !wasHeader ) {
                throw QString( "hmmer_output_header_is_missing" );
            }
            UHMM3SearchCompleteSeqResult& fullSeqRes = res.fullSeqResult;
            readLine( io.get(), buf );
            readLine( io.get(), buf );
            readLine( io.get(), buf );
            readLine( io.get(), buf, &tokens );
            if( buf.startsWith( "[No hits detected" ) ) {
                fullSeqRes.isReported = false;
                break;
            } else {
                fullSeqRes.eval     = getDouble( getNextToken( tokens ) );
                fullSeqRes.score    = getFloat( getNextToken( tokens ) );
                fullSeqRes.bias     = getFloat( getNextToken( tokens ) );
                /* skip best domain res. we will check it later */
                getNextToken( tokens );getNextToken( tokens );getNextToken( tokens );
                fullSeqRes.expectedDomainsNum = getFloat( getNextToken( tokens ) );
                fullSeqRes.reportedDomainsNum = (int)getFloat( getNextToken( tokens ) );
                fullSeqRes.isReported = true;
                wasFullSeqResult = true;
            }
            continue;
        }
        if( buf.startsWith( "Domain and alignment" ) ) {
            if( !wasFullSeqResult ) {
                throw QString( "full_seq_result_is_missing" );
            }
            readLine( io.get(), buf );
            readLine( io.get(), buf );
            readLine( io.get(), buf );
            QList< UHMM3SearchSeqDomainResult >& domainResList = res.domainResList;
            assert( domainResList.isEmpty() );

            int nDomains = res.fullSeqResult.reportedDomainsNum;
            int i = 0;
            for( i = 0; i < nDomains; ++i ) {
                readLine( io.get(), buf, &tokens );
                domainResList << getDomainRes( tokens );
            }
            break;
        }
    } while ( 1 );
    return res;
}

Task::ReportResult GTest_UHMM3SearchCompare::report() {
    assert( !hasErrors() );
    setAndCheckArgs();
    if( hasErrors() ) {
        return ReportResult_Finished;
    }
    
    UHMM3SearchResult trueRes;
    try {
        trueRes = getOriginalSearchResult( trueOutFilename );
    } catch( const QString& ex ) {
        stateInfo.setError( ex );
    } catch(...) {
        stateInfo.setError( "undefined_error_occurred" );
    }

    if( hasErrors() ) {
        return ReportResult_Finished;
    }

    switch( algo ) {
    case GENERAL_SEARCH:
        assert( NULL != generalTask );
        generalCompareResults( generalTask->getResult(), trueRes, stateInfo );
        break;
    case SEQUENCE_WALKER_SEARCH:
        assert( NULL != swTask );
        swCompareResults( swTask->getResults(), trueRes, stateInfo );
        break;
    default:
        assert( 0 && "unknown_algo_type" );
    }
    
    return ReportResult_Finished;
}

} // GB2
