/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "MuscleWorker.h"
#include "MuscleConstants.h"
#include "TaskLocalStorage.h"
#include "MuscleAlignDialogController.h"

#include <workflow/IntegralBusModel.h>
#include <workflow/WorkflowEnv.h>
#include <workflow/WorkflowRegistry.h>
#include <workflow_support/CoreDataTypes.h>
#include <workflow_library/BioDatatypes.h>
#include <workflow_library/BioActorLibrary.h>
#include <workflow_support/DelegateEditors.h>
#include <core_api/Log.h>

/* TRANSLATOR GB2::LocalWorkflow::MuscleWorker */

namespace GB2 {
namespace LocalWorkflow {

static const QString IN_PORT("in");
static const QString OUT_PORT("out");

const QString MuscleWorkerFactory::ACTOR_ID("muscle.align");
const QString MODE_ATTR("mode");
const QString STABLE_ATTR("fix-order");

static LogCategory log(ULOG_CAT_WD);

void MuscleWorkerFactory::init() {
    QList<PortDescriptor*> p; QList<Attribute*> a;
    Descriptor ind(IN_PORT, MuscleWorker::tr("Input MSA"), MuscleWorker::tr("Multiple sequence alignment to be processed."));
    Descriptor oud(OUT_PORT, MuscleWorker::tr("Multiple sequence alignment"), MuscleWorker::tr("Result of alignment."));
    p << new PortDescriptor(ind, BioDataTypes::MULTIPLE_ALIGNMENT_TYPE(), true /*input*/);
    p << new PortDescriptor(oud, BioDataTypes::MULTIPLE_ALIGNMENT_TYPE(), false /*input*/, true /*multi*/);
    Descriptor mod(MODE_ATTR, MuscleWorker::tr("Mode"), 
        MuscleWorker::tr("Selector of preset configurations, that give you the choice of optimizing accuracy, speed, or some compromise between the two. The default favors accuracy."));
    Descriptor sd(STABLE_ATTR, MuscleWorker::tr("Stable order"), 
        MuscleWorker::tr("Do not rearrange aligned sequences (-stable switch of MUSCLE). "
        "<p>Otherwise, MUSCLE re-arranges sequences so that similar sequences are adjacent in the output file. This makes the alignment easier to evaluate by eye. "));
    a << new Attribute(mod, CoreDataTypes::NUM_TYPE(), false, 0);
    a << new Attribute(sd, CoreDataTypes::BOOL_TYPE(), false, true);
    
    Descriptor desc(ACTOR_ID, MuscleWorker::tr("MUSCLE alignment"), 
        MuscleWorker::tr("MUSCLE is public domain multiple alignment software for protein and nucleotide sequences."
        "<p><dfn>MUSCLE stands for MUltiple Sequence Comparison by Log-Expectation.</dfn></p>"));
    ActorPrototype* proto = new BusActorPrototype(desc, p, a);

    QMap<QString, PropertyDelegate*> delegates;    

    {
        QVariantMap vm; 
        vm[DefaultModePreset().name] = 0;
        vm[LargeModePreset().name] = 1;
        vm[RefineModePreset().name] = 2;
        delegates[MODE_ATTR] = new ComboBoxDelegate(vm);
    }

    proto->setEditor(new DelegateEditor(delegates));
    proto->setPrompter(new MusclePrompter());
    proto->setIconPath(":umuscle/images/muscle_16.png");
    WorkflowEnv::getProtoRegistry()->registerProto(BioActorLibrary::CATEGORY_ALIGNMENT(), proto);

    DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById(LocalDomainFactory::ID);
    localDomain->registerEntry(new MuscleWorkerFactory());
}

QString MusclePrompter::composeRichDoc() {
    BusPort* input = qobject_cast<BusPort*>(target->getPort(IN_PORT));
    Actor* producer = input->getProducer(IN_PORT);
    QString producerName = producer ? tr(" from %1").arg(producer->getLabel()) : "";
    QString preset;
    switch (getParameter(MODE_ATTR).toInt()) {
        case 0: preset = DefaultModePreset().name; break;
        case 1: preset = LargeModePreset().name; break;
        case 2: preset = RefineModePreset().name; break;
    }

    QString doc = tr("For each MSA<u>%1</u>, build the alignment using <u>\"%2\" preset</u> and send it to output.")
        .arg(producerName).arg(preset);

    return doc;
}

void MuscleWorker::init() {
    input = ports.value(IN_PORT);
    output = ports.value(OUT_PORT);
    switch (actor->getParameter(MODE_ATTR)->value.toInt()) {
        case 0: DefaultModePreset().apply(cfg); break;
        case 1: LargeModePreset().apply(cfg); break;
        case 2: RefineModePreset().apply(cfg); break;
    }
    cfg.stableMode = actor->getParameter(STABLE_ATTR)->value.toBool();

}

bool MuscleWorker::isReady() {
    return (input && input->hasMessage());
}

Task* MuscleWorker::tick() {
    MAlignment msa = input->get().getData().value<MAlignment>();
    Task* t = new MuscleTask(msa, cfg);
    connect(t, SIGNAL(si_stateChanged()), SLOT(sl_taskFinished()));
    return t;
}

void MuscleWorker::sl_taskFinished() {
    MuscleTask* t = qobject_cast<MuscleTask*>(sender());
    if (t->getState() != Task::State_Finished) return;
    assert(t->resultMA.isNormalized());
    QVariant v = qVariantFromValue<MAlignment>(t->resultMA);
    output->put(Message(BioDataTypes::MULTIPLE_ALIGNMENT_TYPE(), v));
    if (input->isEnded()) {
        output->setEnded();
    }
    log.info(tr("Aligned %1 with MUSCLE").arg(t->resultMA.getName()));
}

bool MuscleWorker::isDone() {
    return !input || input->isEnded();
}

} //namespace LocalWorkflow
} //namespace GB2
