/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "ProjectLoaderImpl.h"

#include <core_api/MainWindow.h>
#include <core_api/Settings.h>
#include <core_api/Log.h>
#include <core_api/ServiceTypes.h>
#include <core_api/AppContext.h>
#include <core_api/ProjectModel.h>

#include <util_gui/DialogUtils.h>

#include <util_gui/CreateDocumentFromTextDialogController.h>

#include "ProjectTasks.h"

#include <QtGui/QAction>

namespace GB2 {

/* TRANSLATOR GB2::ProjectLoaderImpl */

static LogCategory log(ULOG_CAT_CORE_SERVICES);

#define SETTINGS_DIR QString("project_loader/")
#define RECENT_ITEMS_SETTINGS_NAME "recentItems"
#define RECENT_PROJECTS_SETTINGS_NAME "recentProjects"

//////////////////////////////////////////////////////////////////////////
/// ProjectLoaderImpl
//////////////////////////////////////////////////////////////////////////

ProjectLoaderImpl::ProjectLoaderImpl() {
    openProjectAction = newProjectAction = separatorAction1 = separatorAction2 = NULL;
	recentProjectsMenu = NULL;

	assert(AppContext::getProject() == NULL);
	assert(AppContext::getProjectLoader() == NULL);

	ServiceRegistry* sr = AppContext::getServiceRegistry();
	connect(sr, SIGNAL(si_serviceStateChanged(Service*, ServiceState)), SLOT(sl_serviceStateChanged(Service*, ServiceState)));
    
    newProjectAction = new QAction(QIcon(":ugene/images/project_new.png"), tr("project_new"), this);
	newProjectAction->setObjectName(ACTION_PROJECTSUPPORT__NEW_PROJECT);
//    newProjectAction->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_N));
    newProjectAction->setShortcutContext(Qt::WindowShortcut);
    connect(newProjectAction, SIGNAL(triggered()), SLOT(sl_newProject()));

    newDocumentFromtext = new QAction(QIcon(), tr("New document from text"), this);
    newDocumentFromtext->setObjectName("NewDocumentFromText");
    newDocumentFromtext->setShortcutContext(Qt::WindowShortcut);
    connect(newDocumentFromtext, SIGNAL(triggered()), SLOT(sl_newDocumentFromText()));

    openProjectAction = new QAction(QIcon(":ugene/images/project_open.png"), tr("Open"), this);
	openProjectAction->setObjectName(ACTION_PROJECTSUPPORT__OPEN_PROJECT);
    openProjectAction->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_O));
    openProjectAction->setShortcutContext(Qt::WindowShortcut);
    connect(openProjectAction, SIGNAL(triggered()), SLOT(sl_openProject()));

    separatorAction1  = new QAction("-", this);
    separatorAction1->setSeparator(true);

	separatorAction2 = new QAction("-", this);
	separatorAction2->setSeparator(true);

    //add load/close actions to menu and toolbar
    MainWindow* mw = AppContext::getMainWindow();
    MWMenuManager* mm = mw->getMenuManager();
    QMenu* fileMenu = mm->getTopLevelMenu(MWMENU_FILE);

	recentProjectsMenu = new QMenu(tr("Recent Projects"));
	recentProjectsMenu->menuAction()->setObjectName(ACTION_PROJECTSUPPORT__RECENT_PROJECTS_MENU);
	updateRecentProjectsMenu();

    recentItemsMenu = new QMenu(tr("Recent Files"));
    recentItemsMenu->menuAction()->setObjectName("recent_docs_menu_action");
    updateRecentItemsMenu();

	QList<QAction*> actions;
	actions << newProjectAction << openProjectAction << newDocumentFromtext << separatorAction1 <<  recentItemsMenu->menuAction() << recentProjectsMenu->menuAction() << separatorAction2;
	
	fileMenu->insertActions(fileMenu->actions().first(), actions);


	MWToolBarManager* tm = mw->getToolBarManager();
	QToolBar* tb = tm->getToolbar(MWTOOLBAR_MAIN);
	tb->addAction(newProjectAction);
	tb->addAction(openProjectAction);

	updateState();
}

ProjectLoaderImpl::~ProjectLoaderImpl() {
    assert(AppContext::getProject() == NULL);

	delete separatorAction1;
	delete separatorAction2;
    delete openProjectAction;
    delete newProjectAction;
	delete recentProjectsMenu;
	delete recentItemsMenu;

    openProjectAction = newProjectAction = separatorAction1 = separatorAction2 = NULL;
}


void ProjectLoaderImpl::updateState() {
	recentProjectsMenu->setDisabled(recentProjectsMenu->isEmpty());
}

#define MAX_RECENT_FILES 7


void ProjectLoaderImpl::sl_newProject() {
    QWidget *p = (QWidget*)AppContext::getMainWindow()->getQMainWindow();
    ProjectDialogController d(ProjectDialogController::New_Project,p);
	int rc = d.exec();
	AppContext::getSettings()->setValue(SETTINGS_DIR + "last_dir",d.projectFolderEdit->text());

	if (rc == QDialog::Rejected) {
		updateState();
		return;
	}

	QString fileName = d.projectFolderEdit->text() + "/" + d.projectFileEdit->text();
	if (!fileName.endsWith(PROJECTFILE_EXT)) {
		fileName.append(PROJECTFILE_EXT);
	}
	QFileInfo fi(fileName);
	if (fi.exists()) {
		QFile::remove(fileName);
	}

	QString projectName = d.projectNameEdit->text();
	AppContext::getTaskScheduler()->registerTopLevelTask(new OpenProjectTask(fileName, true, projectName));
}

void ProjectLoaderImpl::sl_openProject() {
    LastOpenDirHelper h;
    QString filter = DialogUtils::prepareDocumentsFileFilter(true);
    filter.append("\n"+tr("gb_project_file") + " (*" + PROJECTFILE_EXT + ")");
    QStringList files = QFileDialog::getOpenFileNames(NULL, tr("open_dialog_caption"), h.dir,  filter);
    if (files.isEmpty()) {
        return;
    }
    if (QFileInfo(files.first()).exists()) {
        h.url = files.first();
    }
    QList<QUrl> urls;
    foreach(QString file, files) {
        urls << QUrl("file:///" + file);
    }
    //updateRecentItemsMenu();
    Task* openTask = new OpenProjectTask(urls, true);
	AppContext::getTaskScheduler()->registerTopLevelTask(openTask);	
}


void ProjectLoaderImpl::sl_openRecentFile() {
	QAction *action = qobject_cast<QAction *>(sender());
	assert(action);
	QString url = action->data().toString();
    AppContext::getTaskScheduler()->registerTopLevelTask(new OpenProjectTask(url, true));	
    prependToRecentItems(url);
#ifdef Q_OS_LINUX
    if(QString("4.5.0") == qVersion())
    {
        QTimer::singleShot(0,this,SLOT(sl_updateRecentItemsMenu()));
    }
    else
    {
#endif // Q_OS_LINUX
        updateRecentItemsMenu();
#ifdef Q_OS_LINUX
    }
#endif // Q_OS_LINUX
 }

void ProjectLoaderImpl::prependToRecentProjects(const QString& url) {
    assert(!url.isEmpty());
	QStringList recentFiles =AppContext::getSettings()->getValue(SETTINGS_DIR + RECENT_PROJECTS_SETTINGS_NAME).toStringList();
	recentFiles.removeAll(url);
	recentFiles.prepend(url);
	while(recentFiles.size() > MAX_RECENT_FILES) {
		recentFiles.pop_back();
	}
	AppContext::getSettings()->setValue(SETTINGS_DIR + RECENT_PROJECTS_SETTINGS_NAME, recentFiles);
}

void ProjectLoaderImpl::updateRecentProjectsMenu() {
    assert(recentProjectsMenu!=NULL);
    recentProjectsMenu->clear();
    QStringList recentFiles =AppContext::getSettings()->getValue(SETTINGS_DIR + RECENT_PROJECTS_SETTINGS_NAME).toStringList();
    Project* p = AppContext::getProject();
    foreach (QString f, recentFiles) {
        if ((p==NULL || f != p->getProjectURL()) && !f.isEmpty()) {
            QAction* a = recentProjectsMenu->addAction(f, this, SLOT(sl_openRecentFile()));
            a->setData(f);
            a->setDisabled(!QFile::exists(f));
        }
    }
}


Task* ProjectLoaderImpl::openProjectTask(const QString& file, bool closeActiveProject) {
	return new OpenProjectTask(file, closeActiveProject);
}

Task* ProjectLoaderImpl::openProjectTask(const QList<QUrl>& urls, bool closeActiveProject) {
    return new OpenProjectTask(urls, closeActiveProject);
}

void ProjectLoaderImpl::sl_projectURLChanged(const QString& oldURL) {
    if (!oldURL.isEmpty()) {
        prependToRecentProjects(oldURL);
    }
    rememberProjectURL();
}

void ProjectLoaderImpl::rememberProjectURL() {
    Project* p = AppContext::getProject();
    QString url = p == NULL ? QString() : p->getProjectURL();
    if (!url.isEmpty()) {
        prependToRecentProjects(url);
    }
    updateRecentProjectsMenu();
}

void ProjectLoaderImpl::sl_serviceStateChanged(Service* s, ServiceState prevState) {
    Q_UNUSED(prevState);

	if (s->getType()!=Service_Project) {
		return;
	}
    if (s->isEnabled()) {
        Project* p = AppContext::getProject();
        connect(p, SIGNAL(si_projectURLChanged(const QString&)), SLOT(sl_projectURLChanged(const QString&)));
        connect(p, SIGNAL(si_documentAdded(Document*)), SLOT(sl_documentAdded(Document*)));
    } 
	rememberProjectURL();
    updateState();
}


QString ProjectLoaderImpl::getLastProjectURL() {
    QStringList recentFiles =AppContext::getSettings()->getValue(SETTINGS_DIR + RECENT_PROJECTS_SETTINGS_NAME).toStringList();
    if (!recentFiles.isEmpty()) {
        return recentFiles.first();
    }
    return QString();
}

void ProjectLoaderImpl::prependToRecentItems( const QString& url )
{
    assert(!url.isEmpty());
    QStringList recentFiles =AppContext::getSettings()->getValue(SETTINGS_DIR + RECENT_ITEMS_SETTINGS_NAME).toStringList();
    recentFiles.removeAll(url);
    recentFiles.prepend(url);
    while(recentFiles.size() > MAX_RECENT_FILES) {
        recentFiles.pop_back();
    }
    AppContext::getSettings()->setValue(SETTINGS_DIR + RECENT_ITEMS_SETTINGS_NAME, recentFiles);
    
}

// QT 4.5.0 bug workaround
void ProjectLoaderImpl::sl_updateRecentItemsMenu()
{
    updateRecentItemsMenu();
}

void ProjectLoaderImpl::updateRecentItemsMenu()
{
    assert(recentItemsMenu!=NULL);
    recentItemsMenu->clear();
    QStringList recentFiles =AppContext::getSettings()->getValue(SETTINGS_DIR + RECENT_ITEMS_SETTINGS_NAME).toStringList();
    recentItemsMenu->menuAction()->setEnabled(!recentFiles.isEmpty());
    Project* p = AppContext::getProject();
    foreach (QString f, recentFiles) {
        if ((p==NULL || f != p->getProjectURL()) && !f.isEmpty()) {
            QAction* a = recentItemsMenu->addAction(f, this, SLOT(sl_openRecentFile()));
            a->setData(f);
            a->setDisabled(!QFile::exists(f));
        }
    }

}

void ProjectLoaderImpl::sl_documentAdded( Document* doc )
{
    prependToRecentItems(doc->getURL());
    updateRecentItemsMenu();
}

void ProjectLoaderImpl::sl_newDocumentFromText(){
    QWidget *p = (QWidget*)AppContext::getMainWindow()->getQMainWindow();
    CreateDocumentFromTextDialogController *dialog = new CreateDocumentFromTextDialogController(p);
    if(AppContext::getProject()==NULL){
        AppContext::getTaskScheduler()->registerTopLevelTask(new OpenProjectTaskAndExecDialog(dialog));
    }else{
        dialog->exec();
        delete dialog;
    }
}
//////////////////////////////////////////////////////////////////////////
//ProjectDialogController
//////////////////////////////////////////////////////////////////////////
ProjectDialogController::ProjectDialogController(ProjectDialogController::Mode m, QWidget *p):QDialog(p) {
    setupUi(this);
    fileEditIsEmpty = false;
    QString lastDir =AppContext::getSettings()->getValue(SETTINGS_DIR + "last_dir", QString("")).toString();
    projectFolderEdit->setText(lastDir);

    if (m == Save_Project) {
        setWindowTitle(ProjectLoaderImpl::tr("Save project as"));
        createButton->setText(ProjectLoaderImpl::tr("Save"));
        titleLabel->setVisible(false);
        resize(width(), height() - titleLabel->pixmap()->height());
        frame->setFrameShape(QFrame::NoFrame);
        projectNameEdit->setText(AppContext::getProject()->getProjectName());
        QString url = AppContext::getProject()->getProjectURL();
        if (!url.isEmpty()) {
            QFileInfo fi(url);
            projectFileEdit->setText(fi.completeBaseName());
            projectFolderEdit->setText(fi.absolutePath());
        }
    } else {
        projectNameEdit->setText(ProjectLoaderImpl::tr("new_project_default_name"));
        projectFileEdit->setText(ProjectLoaderImpl::tr("new_project_default_file"));
    }
    projectFolderEdit->setReadOnly(true);
    if (projectFileEdit->text().isEmpty()) {
        fileEditIsEmpty = true;
    }
    connect(folderSelectButton, SIGNAL(clicked()), SLOT(sl_folderSelectClicked()));
    connect(projectFileEdit, SIGNAL(textEdited(const QString&)), SLOT(sl_fileNameEdited(const QString&)));
    connect(projectNameEdit, SIGNAL(textEdited(const QString&)), SLOT(sl_projectNameEdited(const QString&)));
    updateState();
}

void ProjectDialogController::updateState() { 
    bool ready = true;
    
    const QString& folder = projectFolderEdit->text();
    const QString& file = projectFileEdit->text();
    const QString& name = projectNameEdit->text();
    
    //todo: improve check
    if (folder.isEmpty() || file.isEmpty() || name.isEmpty()) {
        ready = false;
    }
    createButton->setEnabled(ready);
}

void ProjectDialogController::keyPressEvent(QKeyEvent* event) {
    int key = event->key();
    if (event->modifiers() == Qt::NoModifier && (key == Qt::Key_Enter || key == Qt::Key_Return)) {
        createButton->animateClick();
    }
    QDialog::keyPressEvent(event);
    
}

void ProjectDialogController::sl_folderSelectClicked() {
    QString folder = QFileDialog::getExistingDirectory();
    if (folder.isEmpty()) return;
    projectFolderEdit->setText(folder);
    updateState();
}

void ProjectDialogController::sl_fileNameEdited(const QString&) {
	//TODO: warn about overwrite
    fileEditIsEmpty = false;
    updateState();
}

void ProjectDialogController::sl_projectNameEdited(const QString& text) {
    if (fileEditIsEmpty) {
        projectFileEdit->setText(text);
    }
    updateState();
    
}


}//namespace

