/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "TaskStatusBar.h"

#include "TaskViewController.h"

#include <core_api/AppContext.h>
#include <core_api/MainWindow.h>

#include <QtCore/QEvent>

#include <QtGui/QPainter>
#include <QtGui/QIcon>

#include <math.h>

namespace GB2 {

TaskStatusBar::TaskStatusBar() {
    nReports = 0;
    tvConnected = false;
    taskToTrack = NULL;
    taskProgressBar = NULL;

    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    l = new QHBoxLayout();
    l->setMargin(0);
    setLayout(l);

    l->addSpacing(20);

    taskInfoLabel = new QLabel();
    taskInfoLabel->setTextFormat(Qt::PlainText);
    taskInfoLabel->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    l->addWidget(taskInfoLabel);

    taskCountLabel = new QLabel();
    taskCountLabel->setTextFormat(Qt::PlainText);
    taskCountLabel->setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Expanding);
    taskCountLabel->setMinimumWidth(100);
    taskCountLabel->setAlignment(Qt::AlignRight | Qt::AlignVCenter);

    l->addWidget(taskCountLabel);

    lampLabel = new QLabel();
    l->addWidget(lampLabel);

#ifdef Q_OS_MAC
    l->addSpacing(16);
#endif

    iconOn = QIcon(":ugene/images/lightbulb.png").pixmap(16, 16);
    iconOff = QIcon(":ugene/images/lightbulb_off.png").pixmap(16, 16);

    connect(AppContext::getTaskScheduler(), SIGNAL(si_stateChanged(Task*)), SLOT(sl_taskStateChanged(Task*)));
    
    lampLabel->installEventFilter(this);
    taskCountLabel->installEventFilter(this);

    updateState();
}

void TaskStatusBar::sl_reportsCountChanged() {
    TaskViewDockWidget* twd = qobject_cast<TaskViewDockWidget*>(sender());
    nReports = twd->countAvailableReports();
    updateState();
}

void TaskStatusBar::updateState() {
    QString reportsString = nReports == 0 ? QString("") : tr("num_reports_%1").arg(nReports);
    if (taskToTrack ==  NULL) {
        taskInfoLabel->setText("");
        if (nReports == 0) {
            taskCountLabel->setText(tr("no_active_tasks"));
        } else {
            taskCountLabel->setText(reportsString);
        }
        lampLabel->setPixmap(iconOff);
        return;
    }
    QString desc = taskToTrack->getStateInfo().getStateDesc();
    QString text = tr("running_task_%1").arg(taskToTrack->getTaskName());
    if (taskToTrack->isCanceled() && !taskToTrack->isFinished()) {
        QString cancelStr = tr("canceling...");
        if (!desc.isEmpty()) {
            cancelStr=", " + cancelStr;
        }
        desc+=cancelStr;
    }
    if (!desc.isEmpty()) {
        text+=tr("_info_%1").arg(desc);
    }
    taskInfoLabel->setText(text);
    int nTasks = AppContext::getTaskScheduler()->getTopLevelTasks().size();
    if (nReports > 0) {
        taskCountLabel->setText(tr("num_tasks_%1_num_reports_%2").arg(nTasks).arg(nReports));
    } else {
        taskCountLabel->setText(tr("num_tasks_%1").arg(nTasks));
    }

    if (taskProgressBar!=NULL) {
        int progress = taskToTrack->getProgress();
        if (progress!=-1) {
            taskProgressBar->setValue(progress);
        }
    }

    lampLabel->setPixmap(iconOn);
}

void TaskStatusBar::sl_taskStateChanged(Task* t) {
    assert(taskToTrack == NULL);
    if (t->isFinished()) {
        return;
    }
    setTaskToTrack(t);
    AppContext::getTaskScheduler()->disconnect(this);
}

void TaskStatusBar::setTaskToTrack(Task* t) { 
    assert(taskToTrack == NULL);
    taskToTrack = t;
    connect(taskToTrack, SIGNAL(si_stateChanged()), SLOT(sl_taskStateChanged()));
    connect(taskToTrack, SIGNAL(si_progressChanged()), SLOT(sl_taskProgressChanged()));
    connect(taskToTrack, SIGNAL(si_descriptionChanged()), SLOT(sl_taskDescChanged()));
    updateState();
}

void TaskStatusBar::sl_taskStateChanged() {
    if (!tvConnected) {
        QWidget* w = AppContext::getMainWindow()->getDockManager()->findWidget(DOCK_TASK_VIEW);
        if (w!=NULL) {
            TaskViewDockWidget* twd = qobject_cast<TaskViewDockWidget*>(w);
            nReports = twd->countAvailableReports();
            connect(twd, SIGNAL(si_reportsCountChanged()), SLOT(sl_reportsCountChanged()));
            tvConnected = true;
        }
    }
    assert(taskToTrack == sender());
    if (!taskToTrack->isFinished()) {
        updateState();
        return;
    }
    taskToTrack->disconnect(this);
    taskToTrack = NULL;
    delete taskProgressBar;
    taskProgressBar= NULL;

    foreach(Task* newT, AppContext::getTaskScheduler()->getTopLevelTasks()) {
        if (!newT->isFinished()) {
            setTaskToTrack(newT);
            break;
        }
    }
    if (taskToTrack == NULL) {
        connect(AppContext::getTaskScheduler(), SIGNAL(si_stateChanged(Task*)), SLOT(sl_taskStateChanged(Task*)));
    }
    updateState();
}

bool TaskStatusBar::eventFilter(QObject *o, QEvent *e) {
    Q_UNUSED(o);
    QEvent::Type t = e->type();
    if (t == QEvent::MouseButtonDblClick) {
        AppContext::getMainWindow()->getDockManager()->toggleDock(DOCK_TASK_VIEW);
    }
    return false;
}

void TaskStatusBar::mouseDoubleClickEvent(QMouseEvent *e) {
    if (taskToTrack!=NULL) {
        QWidget* w = AppContext::getMainWindow()->getDockManager()->activateDock(DOCK_TASK_VIEW);
        if (w!=NULL) {
            TaskViewDockWidget* twd = qobject_cast<TaskViewDockWidget*>(w);
            twd->selectTask(taskToTrack);
        }
    }
    QWidget::mouseDoubleClickEvent(e);
}


void TaskStatusBar::sl_taskProgressChanged() {
    assert(taskToTrack == sender());

    if (taskProgressBar==NULL) {
        taskProgressBar = new QProgressBar();
        taskProgressBar->setRange(0, 100);
        taskProgressBar->setValue(taskToTrack->getProgress());
        taskProgressBar->setFixedWidth(120);
        taskProgressBar->setFixedHeight(16);
    
        l->insertWidget(2, taskProgressBar);
    }
    
    updateState();
}

void TaskStatusBar::sl_taskDescChanged() {
    assert(taskToTrack == sender());
    updateState();
}

void TaskStatusBar::drawProgress(QLabel* label) {
    static QColor piecolor("#fdc689");
    
    int percent = taskToTrack->getStateInfo().progress;
    int h = height()-2;
    //float radius = h / 2;
    QPixmap pix(h, h);
    QPainter p(&pix);

    p.fillRect(pix.rect(), palette().window().color());
    
    p.setPen(piecolor);
    p.setBrush(piecolor);
    p.drawPie(pix.rect(), -90, qRound(- percent * 57.60));

    p.setPen(Qt::black);
    p.drawText(pix.rect(), Qt::AlignCenter | Qt::TextDontClip, QString("%1").arg(percent));

    label->setPixmap(pix);
}

} //namespace
