/*
 * Copyright (C) 2012 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by David Calle <davidc@framli.eu>
 *             Michal Hruby <michal.hruby@canonical.com>
 *
 */

using Dee;
using Gee;

namespace Unity.MusicLens
{

    private enum Columns
    {
        URI,
        TITLE,
        ARTIST,
        ALBUM,
        ARTWORK,
        MIMETYPE,
        GENRE,
        ALBUM_ARTIST,
        TRACK_NUMBER,
        YEAR,
        PLAY_COUNT,

        N_COLUMNS
    }

    class RhythmboxCollection : Object
    {

        SequenceModel all_tracks;
        FilterModel tracks_by_play_count;

        HashTable<unowned string, Variant> variant_store;
        HashTable<int, Variant> int_variant_store;
        Variant row_buffer[11];

        Analyzer analyzer;
        Index index;
        ICUTermFilter ascii_filter;

        string media_art_dir;

        class XmlParser: Object
        {
          const MarkupParser parser =
          {
            start_tag,
            end_tag,
            process_text,
            null,
            null
          };

          // contains genre maps
          Genre genre = new Genre ();

          MarkupParseContext context;
          bool is_rhythmdb_xml = false;

          construct
          {
            context = new MarkupParseContext (parser, 0, this, null);
          }

          public bool parse (string content, size_t len) throws MarkupError
          {
            return context.parse (content, (ssize_t) len);
          }

          bool processing_song;
          Track current_track;
          int current_data = -1;

          private void start_tag (MarkupParseContext context, string name,
            [CCode (array_length = false, array_null_terminated = true)] string[] attr_names, [CCode (array_length = false, array_null_terminated = true)] string[] attr_values)
            throws MarkupError
          {
            if (!processing_song)
            {
              switch (name)
              {
                case "rhythmdb": is_rhythmdb_xml = true; break;
                case "entry":
                  bool is_song = false;
                  for (int i = 0; attr_names[i] != null; i++)
                  {
                    if (attr_names[i] == "type" && attr_values[i] == "song")
                      is_song = true;
                  }
                  if (!is_song) return;
                  processing_song = true;
                  current_track = new Track ();
                  break;
              }
            }
            else
            {
              switch (name)
              {
                case "location": current_data = Columns.URI; break;
                case "title": current_data = Columns.TITLE; break;
                case "artist": current_data = Columns.ARTIST; break;
                case "album": current_data = Columns.ALBUM; break;
                case "genre": current_data = Columns.GENRE; break;
                case "track-number": current_data = Columns.TRACK_NUMBER; break;
                case "play-count": current_data = Columns.PLAY_COUNT; break;
                case "date": current_data = Columns.YEAR; break;
                case "media-type": current_data = Columns.MIMETYPE; break;
                case "album-artist": current_data = Columns.ALBUM_ARTIST; break;
                default: current_data = -1; break;
              }
            }
          }

          public signal void track_info_ready (Track track);

          private void end_tag (MarkupParseContext content, string name)
            throws MarkupError
          {
            switch (name)
            {
              case "location":
              case "title":
              case "artist":
              case "album":
              case "genre":
              case "track-number":
              case "play-count":
              case "date":
              case "media-type":
              case "album-artist":
                if (current_data >= 0) current_data = -1;
                break;
              case "entry":
                if (processing_song && current_track != null)
                {
                  track_info_ready (current_track);
                }
                processing_song = false;
                break;
            }
          }

          private void process_text (MarkupParseContext context,
                                     string text, size_t text_len)
            throws MarkupError
          {
            if (!processing_song || current_data < 0) return;
            switch (current_data)
            {
              case Columns.URI: current_track.uri = text; break;
              case Columns.TITLE: current_track.title = text; break;
              case Columns.ARTIST: current_track.artist = text; break;
              case Columns.ALBUM: current_track.album = text; break;
              case Columns.ALBUM_ARTIST: 
                current_track.album_artist = text;
                break;
              case Columns.GENRE:
                current_track.genre = genre.get_id_for_genre (text.down ());
                break;
              case Columns.MIMETYPE:
                current_track.mime_type = text;
                break;
              case Columns.YEAR:
                current_track.year = int.parse (text) / 365;
                break;
              case Columns.PLAY_COUNT:
                current_track.play_count = int.parse (text);
                break;
              case Columns.TRACK_NUMBER:
                current_track.track_number = int.parse (text);
                break;
            }
          }
        }

        construct
        {
          static_assert (11 == Columns.N_COLUMNS); // sync with row_buffer size
          media_art_dir = Path.build_filename (
              Environment.get_user_cache_dir (), "media-art");

          variant_store = new HashTable<unowned string, Variant> (str_hash,
                                                                  str_equal);
          int_variant_store = new HashTable<int, Variant> (direct_hash,
                                                           direct_equal);
          all_tracks = new SequenceModel ();
          // the columns correspond to the Columns enum
          all_tracks.set_schema ("s", "s", "s", "s", "s", "s", "s", "s", "i", "i", "i");
          assert (all_tracks.get_schema ().length == Columns.N_COLUMNS);
          

          var filter = Dee.Filter.new_sort ((row1, row2) =>
          {
            int a = row1[Columns.PLAY_COUNT].get_int32 ();
            int b = row2[Columns.PLAY_COUNT].get_int32 ();

            return b - a; // higher play count first
          });
          tracks_by_play_count = new FilterModel (all_tracks, filter);

          ascii_filter = new ICUTermFilter.ascii_folder ();
          analyzer = new TextAnalyzer ();
          analyzer.add_term_filter ((terms_in, terms_out) =>
          {
            foreach (unowned string term in terms_in)
            {
              var folded = ascii_filter.apply (term);
              terms_out.add_term (term);
              if (folded != term) terms_out.add_term (folded);
            }
          });
          var reader = ModelReader.new ((model, iter) =>
          {
            var s ="%s\n%s\n%s".printf (model.get_string (iter, Columns.TITLE),
                                        model.get_string (iter, Columns.ARTIST),
                                        model.get_string (iter, Columns.ALBUM));
            return s;
          });

          index = new TreeIndex (all_tracks, analyzer, reader);
        }
        
        private string? get_albumart (Track track)
        {
            var artist = track.album_artist ?? track.artist;
            var album = track.album;

            var artist_norm = artist.normalize (-1, NormalizeMode.NFKD);
            var album_norm = album.normalize (-1, NormalizeMode.NFKD);

            var artist_md5 = Checksum.compute_for_string (ChecksumType.MD5,
                                                          artist_norm);
            var album_md5 = Checksum.compute_for_string (ChecksumType.MD5,
                                                         album_norm);

            string filename;
            filename = Path.build_filename (media_art_dir,
                "album-%s-%s".printf (artist_md5, album_md5));
            if (FileUtils.test (filename, FileTest.EXISTS)) return filename;

            var combined = "%s\t%s".printf (artist, album).normalize (-1, NormalizeMode.NFKD);
            filename = Path.build_filename (media_art_dir,
                "album-%s.jpg".printf (Checksum.compute_for_string (
                    ChecksumType.MD5, combined)));
            if (FileUtils.test (filename, FileTest.EXISTS)) return filename;

            // Try Nautilus thumbnails
            try
            {
                File artwork_file = File.new_for_uri (track.uri);
                var info = artwork_file.query_info (FILE_ATTRIBUTE_THUMBNAIL_PATH, 0, null);
                var thumbnail_path = info.get_attribute_string (FILE_ATTRIBUTE_THUMBNAIL_PATH);
                if (thumbnail_path != null) return thumbnail_path;
            } catch {}

            // Try covers folder
            string artwork = Path.build_filename (
                Environment.get_user_cache_dir (), "rhythmbox", "covers",
                "%s - %s.jpg".printf (track.artist, track.album));
            if (FileUtils.test (artwork, FileTest.EXISTS)) return artwork;

            return null;
        }

        private Variant cached_variant_for_string (string? input)
        {
          unowned string text = input != null ? input : "";
          Variant? v = variant_store[text];
          if (v != null) return v;

          v = new Variant.string (text);
          // key is owned by value... awesome right?
          variant_store[v.get_string ()] = v;
          return v;
        }

        private Variant cached_variant_for_int (int input)
        {
          Variant? v = int_variant_store[input];
          if (v != null) return v;

          v = new Variant.int32 (input);
          // let's not cache every random integer
          if (input < 128)
            int_variant_store[input] = v;
          return v;
        }

        private void prepare_row_buffer (Track track)
        {
          Variant uri = new Variant.string (track.uri);
          Variant title = new Variant.string (track.title);
          Variant artist = cached_variant_for_string (track.artist);
          Variant album_artist = cached_variant_for_string (track.album_artist);
          Variant album = cached_variant_for_string (track.album);
          Variant mime_type = cached_variant_for_string (track.mime_type);
          Variant artwork = cached_variant_for_string (track.artwork_path);
          Variant genre = cached_variant_for_string (track.genre);
          Variant track_number = cached_variant_for_int (track.track_number);
          Variant year = cached_variant_for_int (track.year);
          Variant play_count = cached_variant_for_int (track.play_count);

          row_buffer[0] = uri;
          row_buffer[1] = title;
          row_buffer[2] = artist;
          row_buffer[3] = album;
          row_buffer[4] = artwork;
          row_buffer[5] = mime_type;
          row_buffer[6] = genre;
          row_buffer[7] = album_artist;
          row_buffer[8] = track_number;
          row_buffer[9] = year;
          row_buffer[10] = play_count;
        }

        public void parse_file (string path)
        {
          var parser = new XmlParser ();
          parser.track_info_ready.connect ((track) =>
          {
            // Get cover art
            string albumart = get_albumart (track);
            if (albumart != null)
              track.artwork_path = albumart;
            else
              track.artwork_path = "audio-x-generic";

            prepare_row_buffer (track);
            all_tracks.append_row (row_buffer);
          });

          var file = File.new_for_path (path);

          try
          {
            var stream = file.read (null);
            uint8 buffer[65536];

            size_t bytes_read;
            while ((bytes_read = stream.read (buffer, null)) > 0)
            {
              parser.parse ((string) buffer, bytes_read);
            }
          }
          catch (Error err)
          {
            warning ("Error while parsing rhythmbox DB: %s", err.message);
          }
        }

        public void search (LensSearch search, 
                            SearchType search_type, 
                            GLib.List<FilterParser>? filters = null,
                            int max_results = -1,
                            int category_override = -1)
        {
            int num_results = 0;
            var empty_search = search.search_string.strip () == "";
            int min_year;
            int max_year;
            int category_id;

            Model model = all_tracks;
            get_decade_filter (filters, out min_year, out max_year);
            var active_genres = get_genre_filter (filters);

            if (empty_search)
            {
                // display a couple of most played songs
                model = tracks_by_play_count;
                var iter = model.get_first_iter ();
                var end_iter = model.get_last_iter ();
                var albums_list_nosearch = new HashSet<string> ();

                while (iter != end_iter)
                {
                    int year = model.get_int32 (iter, Columns.YEAR);
                    unowned string genre = model.get_string (iter, Columns.GENRE);

                    // check filters
                    if (year < min_year || year > max_year)
                    {
                        iter = model.next (iter);
                        continue;
                    }
                    
                    // check filters
                    if (active_genres != null) {
                        if (!(genre in active_genres)) {
                            iter = model.next (iter);
                            continue;
                        }
                    }

                    unowned string album = model.get_string (iter,
                                                             Columns.ALBUM);
                    // it's not first as in track #1, but first found from album
                    bool first_track_from_album = !(album in albums_list_nosearch);
                    albums_list_nosearch.add (album);
                    
                    if (first_track_from_album)
                    {
                        category_id = category_override >= 0 ?
                            category_override : Category.ALBUMS;

                        search.results_model.append (
                            model.get_string (iter, Columns.URI),
                            model.get_string (iter, Columns.ARTWORK),
                            category_id,
                            model.get_string (iter, Columns.MIMETYPE),
                            model.get_string (iter, Columns.ALBUM),
                            model.get_string (iter, Columns.ARTIST),
                            model.get_string (iter, Columns.URI));
                    }

                    category_id = category_override >= 0 ?
                        category_override : Category.SONGS;

                    search.results_model.append (
                        model.get_string (iter, Columns.URI),
                        model.get_string (iter, Columns.ARTWORK),
                        category_id,
                        model.get_string (iter, Columns.MIMETYPE),
                        model.get_string (iter, Columns.TITLE),
                        model.get_string (iter, Columns.ARTIST),
                        model.get_string (iter, Columns.URI));

                    num_results++;
                    if (max_results >= 0 && num_results >= max_results) break;

                    iter = model.next (iter);
                }
                return;
            }


            var term_list = Object.new (typeof (Dee.TermList)) as Dee.TermList;
            // search only the folded terms, FIXME: is that a good idea?
            analyzer.tokenize (ascii_filter.apply (search.search_string),
                               term_list);

            var matches = new Sequence<Dee.ModelIter> ();
            bool first_pass = true;
            foreach (unowned string term in term_list)
            {
                // FIXME: use PREFIX search only for the last term?
                var result_set = index.lookup (term, TermMatchFlag.PREFIX);

                CompareDataFunc<Dee.ModelIter> cmp_func = (a, b) =>
                {
                    return a == b ? 0 : ((void*) a > (void*) b ? 1 : -1);
                };

                // intersect the results (cause we want to AND the terms)
                var remaining = new Sequence<Dee.ModelIter> ();
                foreach (var item in result_set)
                {
                    if (first_pass)
                        matches.insert_sorted (item, cmp_func);
                    else if (matches.lookup (item, cmp_func) != null)
                        remaining.insert_sorted (item, cmp_func);
                }
                if (!first_pass) matches = (owned) remaining;
                // final result set empty already?
                if (matches.get_begin_iter () == matches.get_end_iter ()) break;

                first_pass = false;
            }

            // matches now contain iterators into the all_tracks model which
            // match the search string
            var seq_iter = matches.get_begin_iter ();
            var seq_end_iter = matches.get_end_iter ();
            
            var albums_list = new HashSet<string> ();
            while (seq_iter != seq_end_iter)
            {
                var model_iter = seq_iter.get ();
                int year = model.get_int32 (model_iter, Columns.YEAR);
                string genre = model.get_string (model_iter, Columns.GENRE);

                // check filters
                if (year < min_year || year > max_year)
                {
                    seq_iter = seq_iter.next ();
                    continue;
                }
                
                // check filters
                if (active_genres != null) {
                    bool genre_match = (genre in active_genres);
                    if (!genre_match) {
                        seq_iter = seq_iter.next ();
                        continue;
                    }
                }

                unowned string album = model.get_string (model_iter,
                                                         Columns.ALBUM);
                // it's not first as in track #1, but first found from album
                bool first_track_from_album = !(album in albums_list);
                albums_list.add (album);

                if (first_track_from_album)
                {
                    category_id = category_override >= 0 ?
                        category_override : Category.ALBUMS;

                    search.results_model.append (
                        model.get_string (model_iter, Columns.URI),
                        model.get_string (model_iter, Columns.ARTWORK),
                        category_id,
                        model.get_string (model_iter, Columns.MIMETYPE),
                        model.get_string (model_iter, Columns.ALBUM),
                        model.get_string (model_iter, Columns.ARTIST),
                        model.get_string (model_iter, Columns.URI));
                }

                category_id = category_override >= 0 ?
                    category_override : Category.SONGS;

                search.results_model.append (
                    model.get_string (model_iter, Columns.URI),
                    model.get_string (model_iter, Columns.ARTWORK),
                    category_id,
                    model.get_string (model_iter, Columns.MIMETYPE),
                    model.get_string (model_iter, Columns.TITLE),
                    model.get_string (model_iter, Columns.ARTIST),
                    model.get_string (model_iter, Columns.URI));

                num_results++;
                if (max_results >= 0 && num_results >= max_results) break;

                seq_iter = seq_iter.next ();
            }
        }

        private void get_decade_filter (GLib.List<FilterParser> filters,
                                        out int min_year, out int max_year)
        {
            Filter? filter = null;
            foreach (var parser in filters)
            {
                if (parser is DecadeFilterParser) filter = parser.filter;
            }

            if (filter == null || !filter.filtering)
            {
                min_year = 0;
                max_year = int.MAX;
                return;
            }

            var mrf = filter as MultiRangeFilter;
            min_year = int.parse (mrf.get_first_active ().id);
            // it's supposed to be a decade, so 2000-2009
            max_year = int.parse (mrf.get_last_active ().id) + 9;
        }
        
        private Set<string>? get_genre_filter (GLib.List<FilterParser> filters)
        {
            Filter? filter = null;
            foreach (var parser in filters)
            {
                if (parser is GenreFilterParser) filter = parser.filter;
            }
            if (filter == null || !filter.filtering)
            {
                return null;
            }

            var active_genres = new HashSet<string> ();
            var all_genres = filter as CheckOptionFilterCompact;
            foreach (FilterOption option in all_genres.options)
            {
                if (option.id == null || !option.active) continue;
                active_genres.add (option.id);
            }

            return active_genres;
        }
    }
}
