/**
 * geis.h
 *
 * Copyright 2010 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.	 See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin St, Fifth Floor, Boston, MA  02110-1301 US
 */
#ifndef GEIS_GEIS_H_
#define GEIS_GEIS_H_

#define GEIS_VERSION_1_0  1

#include <geis/geisimpl.h>

/* Standard fundamental gestures */
#define GEIS_GESTURE_DRAG    "Drag"
#define GEIS_GESTURE_PINCH   "Pinch"
#define GEIS_GESTURE_ROTATE  "Rotate"
#define GEIS_GESTURE_TAP     "Tap"

/* Gesture names for the Simplified Interface */
#define GEIS_GESTURE_TYPE_DRAG1   "Drag,touch=1"
#define GEIS_GESTURE_TYPE_DRAG2   "Drag,touch=2"
#define GEIS_GESTURE_TYPE_DRAG3   "Drag,touch=3"
#define GEIS_GESTURE_TYPE_DRAG4   "Drag,touch=4"
#define GEIS_GESTURE_TYPE_DRAG5   "Drag,touch=5"
#define GEIS_GESTURE_TYPE_PINCH1  "Pinch,touch=1"
#define GEIS_GESTURE_TYPE_PINCH2  "Pinch,touch=2"
#define GEIS_GESTURE_TYPE_PINCH3  "Pinch,touch=3"
#define GEIS_GESTURE_TYPE_PINCH4  "Pinch,touch=4"
#define GEIS_GESTURE_TYPE_PINCH5  "Pinch,touch=5"
#define GEIS_GESTURE_TYPE_ROTATE1 "Rotate,touch=1"
#define GEIS_GESTURE_TYPE_ROTATE2 "Rotate,touch=2"
#define GEIS_GESTURE_TYPE_ROTATE3 "Rotate,touch=3"
#define GEIS_GESTURE_TYPE_ROTATE4 "Rotate,touch=4"
#define GEIS_GESTURE_TYPE_ROTATE5 "Rotate,touch=5"
#define GEIS_GESTURE_TYPE_TAP1    "Tap,touch=1"
#define GEIS_GESTURE_TYPE_TAP2    "Tap,touch=2"
#define GEIS_GESTURE_TYPE_TAP3    "Tap,touch=3"
#define GEIS_GESTURE_TYPE_TAP4    "Tap,touch=4"
#define GEIS_GESTURE_TYPE_TAP5    "Tap,touch=5"

/* Standard fundamental gesture attributes */
#define GEIS_GESTURE_ATTRIBUTE_ANGLE            "angle"
#define GEIS_GESTURE_ATTRIBUTE_ANGLE_DELTA      "angle delta"
#define GEIS_GESTURE_ATTRIBUTE_ANGULAR_VELOCITY "angular velocity"
#define GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_X1   "boundingbox x1"
#define GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_Y1   "boundingbox y1"
#define GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_X2   "boundingbox x2"
#define GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_Y2   "boundingbox y2"
#define GEIS_GESTURE_ATTRIBUTE_CHILD_WINDOW_ID  "child window id"
#define GEIS_GESTURE_ATTRIBUTE_DELTA_X          "delta x"
#define GEIS_GESTURE_ATTRIBUTE_DELTA_Y          "delta y"
#define GEIS_GESTURE_ATTRIBUTE_DEVICE_ID        "device id"
#define GEIS_GESTURE_ATTRIBUTE_EVENT_WINDOW_ID  "event window id"
#define GEIS_GESTURE_ATTRIBUTE_FOCUS_X          "focus x"
#define GEIS_GESTURE_ATTRIBUTE_FOCUS_Y          "focus y"
#define GEIS_GESTURE_ATTRIBUTE_GESTURE_NAME     "gesture name"
#define GEIS_GESTURE_ATTRIBUTE_POSITION_X       "position x"
#define GEIS_GESTURE_ATTRIBUTE_POSITION_Y       "position y"
#define GEIS_GESTURE_ATTRIBUTE_RADIAL_VELOCITY  "radial velocity"
#define GEIS_GESTURE_ATTRIBUTE_RADIUS_DELTA     "radius delta"
#define GEIS_GESTURE_ATTRIBUTE_RADIUS           "radius"
#define GEIS_GESTURE_ATTRIBUTE_ROOT_WINDOW_ID   "root window id"
#define GEIS_GESTURE_ATTRIBUTE_TAP_TIME         "tap time"
#define GEIS_GESTURE_ATTRIBUTE_TIMESTAMP        "timestamp"
#define GEIS_GESTURE_ATTRIBUTE_TOUCHES          "touches"
#define GEIS_GESTURE_ATTRIBUTE_VELOCITY_X       "velocity x"
#define GEIS_GESTURE_ATTRIBUTE_VELOCITY_Y       "velocity y"
#define GEIS_GESTURE_ATTRIBUTE_TOUCH_0_ID       "touch 0 id"
#define GEIS_GESTURE_ATTRIBUTE_TOUCH_0_X        "touch 0 x"
#define GEIS_GESTURE_ATTRIBUTE_TOUCH_0_Y        "touch 0 y"
#define GEIS_GESTURE_ATTRIBUTE_TOUCH_1_ID       "touch 1 id"
#define GEIS_GESTURE_ATTRIBUTE_TOUCH_1_X        "touch 1 x"
#define GEIS_GESTURE_ATTRIBUTE_TOUCH_1_Y        "touch 1 y"
#define GEIS_GESTURE_ATTRIBUTE_TOUCH_2_ID       "touch 2 id"
#define GEIS_GESTURE_ATTRIBUTE_TOUCH_2_X        "touch 2 x"
#define GEIS_GESTURE_ATTRIBUTE_TOUCH_2_Y        "touch 2 y"
#define GEIS_GESTURE_ATTRIBUTE_TOUCH_3_ID       "touch 3 id"
#define GEIS_GESTURE_ATTRIBUTE_TOUCH_3_X        "touch 3 x"
#define GEIS_GESTURE_ATTRIBUTE_TOUCH_3_Y        "touch 3 y"
#define GEIS_GESTURE_ATTRIBUTE_TOUCH_4_ID       "touch 4 id"
#define GEIS_GESTURE_ATTRIBUTE_TOUCH_4_X        "touch 4 x"
#define GEIS_GESTURE_ATTRIBUTE_TOUCH_4_Y        "touch 4 y"

/**
 * @defgroup geis_status Status and Errors
 *
 * Most GEIS API calls return a status code indicating success or, in the event
 * of a lack of success, the reson for failure.
 *
 * @{
 */

/**
 * Errors returned from calls.
 */
typedef enum GeisStatus
{
  GEIS_STATUS_SUCCESS       = 0,    /**< normal successful completion */
  GEIS_STATUS_NOT_SUPPORTED = 10,   /**< a requested feature is not supported */
  GEIS_BAD_ARGUMENT         = 1000, /**< a bad argument value was passed */
  GEIS_UNKNOWN_ERROR        = 9999  /**< any other error condition */
} GeisStatus;

/* @} */

/**
 * @defgroup geis_meta Initialization and Cleanup
 *
 * Each instance of a gesture subscription must be created using the geis_init()
 * call and destroyed using the geis_finish() call.
 *
 * A particular subscription instance is associated with a display region.  The
 * nature of the display region depends on the underlying display technology.
 * For example, an X11 window or even a subregion of an X11 window could be an
 * associated display region when geis is layered over X11 technology.
 *
 * The nature of the display desciption information depends on the actual
 * underlyinggeis implementation and is documented separately.  The
 * implementation-specific description must be passed to geis_init using a 
 * GeisWinInfo structure.
 *
 * @{
 */

/**
 * An opaque type representing a geis gesture subscription instance.
 */
typedef struct _GeisInstance *GeisInstance;

/**
 * @struct GeisWinInfo
 * Generic display region description block
 */
typedef struct GeisWinInfo
{
  uint32_t win_type;
  void    *win_info;
} GeisWinInfo;

/**
 * Initializes a geis subscription instance for a display region.
 *
 * @param[in]  win_info         a display region description block
 *                              -- see geis implementtaion documentation
 * @param[out] geis_instance    an opaque pointer to a geis gesture subscription
 *                              instance
 *
 * @retval GEIS_BAD_ARGUMENT    an invalid GeisWinInfo was passed
 * @retval GEIS_STATUS_SUCCESS  normal successful completion
 */
GEIS_API GeisStatus geis_init(GeisWinInfo *win_info, GeisInstance *geis_instance);

/**
 * Cleans up a geis subscriotion instance for a display region.
 *
 * @param[in] geis_instance     an opaque pointer to a geis gesture subscription
 *                              instance
 *
 * @retval GEIS_BAD_ARGUMENT    an invalid GeisInstance was passed
 * @retval GEIS_STATUS_SUCCESS  normal successful completion
 */
GEIS_API GeisStatus geis_finish(GeisInstance  geis_instance);

/* @} */

/**
 * @defgroup geis_config Configuration and Control
 * @{
 */

#define GEIS_CONFIG_UNIX_FD 10001

/**
 * Indicates if a particular feaure is supported.
 *
 * @param[in] geis_instance     an opaque pointer to a geis gesture subscription
 *                              instance
 *
 * @retval GEIS_BAD_ARGUMENT    an invalid argument value was passed
 * @retval GEIS_STATUS_SUCCESS  normal successful completion
 */
GEIS_API GeisStatus geis_configuration_supported(GeisInstance geis_instance,
                                                 int          configuration_item);

/**
 * Gets a feature configuration value.
 *
 * @param[in] geis_instance     an opaque pointer to a geis gesture subscription
 *                              instance
 *
 * @retval GEIS_BAD_ARGUMENT    an invalid argument value was passed
 * @retval GEIS_STATUS_SUCCESS  normal successful completion
 */
GEIS_API GeisStatus geis_configuration_get_value(GeisInstance geis_instance, 
                                                 int          configuration_item,
                                                 void         *value);

/**
 * Sets a feature configuration value.
 *
 * @param[in] geis_instance     an opaque pointer to a geis gesture subscription
 *                              instance
 *
 * @retval GEIS_BAD_ARGUMENT    an invalid argument value was passed
 * @retval GEIS_STATUS_SUCCESS  normal successful completion
 */
GEIS_API GeisStatus geis_configuration_set_value(GeisInstance geis_instance,
                                                 int          configuration_item, 
                                                 void         *value);

/**
 * Dispatches geis events until there are no further events available.
 *
 * @param[in] geis_instance     an opaque pointer to a geis gesture subscription
 *                              instance
 *
 * This function is used to integrate geis even dispatch into the main event
 * loop of an application or toolkit.
 *
 * @retval GEIS_BAD_ARGUMENT    an invalid GeisInstance was passed
 * @retval GEIS_STATUS_SUCCESS  normal successful completion
 */
GEIS_API GeisStatus geis_event_dispatch(GeisInstance geis_instance);

/* @} */

typedef enum GeisAttrType
{
  GEIS_ATTR_TYPE_UNKNOWN,
  GEIS_ATTR_TYPE_BOOLEAN,
  GEIS_ATTR_TYPE_FLOAT,
  GEIS_ATTR_TYPE_INTEGER,
  GEIS_ATTR_TYPE_STRING
} GeisAttrType;

/**
 * @defgroup geis_input Input Devices
 * @{
 */

typedef unsigned int GeisInputDeviceId;

#define GEIS_ALL_INPUT_DEVICES ((GeisInputDeviceId)0)

/**
 * Prototype for input device callback functions.
 */
typedef void (*GeisInputCallback)(void             *cookie,
                                  GeisInputDeviceId device_id,
                                  void             *attrs);

/**
 * Callback functions used to handle changes in the available input devices.
 */
typedef struct GeisInputFuncs
{
  GeisInputCallback  added;   /**< Receives new input device notices */
  GeisInputCallback  changed; /**< Receives modified input device notices */
  GeisInputCallback  removed; /**< Receives removes input device notices */

} GeisInputFuncs;


/**
 * Registers a callback to receive information on input devices.
 *
 * @param[in] geis_instance   points to a geis gesture subscription
 *                            instance
 * @param[in] funcs           points to a GeisInputFuncs table
 * @param[in] cookie          an application specific value to be passed to
 *                            the callback
 *
 * The callback is called for each gesture-capable input device available for
 * the display region associated with the geis subscription instance.  Over
 * time, as gesture-capable input devices appear and disappear or change their
 * abilities or configuration, the callback may be called again.
 *
 * @retval GEIS_BAD_ARGUMENT    an invalid argument value was passed
 * @retval GEIS_STATUS_SUCCESS  normal successful completion
 */
GEIS_API GeisStatus geis_input_devices(GeisInstance    geis_instance,
                                       GeisInputFuncs *func,
                                       void           *cookie);

/* @} */

/**
 * @defgroup geis_subscription Gesture Subscription
 * @{
 */
typedef unsigned int GeisGestureType;
typedef unsigned int GeisGestureId;

#define GEIS_ALL_GESTURES ((GeisGestureType)0)
#define GEIS_NO_GESTURE_ID ((GeisGestureId)0)

/**
 * Gesture Attributes
 */
typedef struct GeisGestureAttr
{
  GeisString name;
  GeisAttrType type;
  union
  {
    GeisBoolean boolean_val;
    GeisFloat   float_val;
    GeisInteger integer_val;
    GeisString  string_val;
  };
} GeisGestureAttr;

/**
 * A callback used for different gesture events.
 *
 * @param[in] cookie         an application-specific value to be passed to the
 *                           callback.
 * @param[in] gesture_type   a gesture type
 * @param[in] gesture_id     a unique gesture identifier
 * @param[in] attrs          parameters
 */
typedef void (*GeisGestureCallback)(void             *cookie,
                                    GeisGestureType   gesture_type,
                                    GeisGestureId     gesture_id,
                                    GeisSize          attr_count,
                                    GeisGestureAttr  *attrs);

/**
 * A structure of callback functions.
 */
typedef struct GeisGestureFuncs
{
  GeisGestureCallback  added;
  GeisGestureCallback  removed;
  GeisGestureCallback  start;
  GeisGestureCallback  update;
  GeisGestureCallback  finish;
} GeisGestureFuncs;


/**
 * Registers a callback to receive gesture events.
 *
 * @param[in] geis_instance     an opaque pointer to a geis gesture subscription
 *                              instance
 * @param[in] input_list        a null-terminated list of input device IDs
 * @param[in] gesture_list      a null-terminated list of C-style strings naming
 *                              gestures for subscription
 * @param[in] funcs             a pointer to a GeisGestureFuncs structure
 * @param[in] cookie            an application specific value to be passed to
 *                              the callback
 *
 * @retval GEIS_BAD_ARGUMENT    an invalid argument value was passed
 * @retval GEIS_STATUS_SUCCESS  normal successful completion
 */
GEIS_API GeisStatus geis_subscribe(GeisInstance         geis_instance,
                                   GeisInputDeviceId   *input_list,
                                   const char*         *gesture_list,
                                   GeisGestureFuncs    *funcs,
                                   void                *cookie);

/**
 * Unsubscribes to one or more gestures.
 *
 * @param[in] geis_instance     an opaque pointer to a geis gesture subscription
 *                              instance
 * @param[in] gesture_list      a null-terminated list of gesture types
 */
GEIS_API GeisStatus geis_unsubscribe(GeisInstance     geis_instance,
                                     GeisGestureType *gesture_list);

/* @} */

#endif /* GEIS_GEIS_H_ */
