/*
TRANSLATOR Panoramio::Item
*/
/*
TRANSLATOR Panoramio::Core
*/
/*
TRANSLATOR Panoramio::GoogleDialog
*/
/*
TRANSLATOR Panoramio::DialogWidget
*/

/*
 * Wally - Qt4 wallpaper/background changer
 * Copyright (C) 2009  Antonio Di Monaco <tony@becrux.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <QtCore>
#include <QtGui>
#include <QtNetwork>
#include <QtScript>

#include "gui.h"
#include "httpengine.h"
#include "panoramio.h"
#include "mapviewer.h"

using namespace Panoramio;

Item::Item() : HttpEngine::Item()
{
  _requestedSize = Original;
  _searchOrder = Popularity;
  _minLatitude = _maxLatitude = _minLongitude = _maxLongitude = 0.0;
}

Item::Item(Item *item) : HttpEngine::Item(item)
{
  _requestedSize = item->_requestedSize;
  _searchOrder = item->_searchOrder;
  _minLatitude = item->_minLatitude;
  _maxLatitude = item->_maxLatitude;
  _minLongitude = item->_minLongitude;
  _maxLongitude = item->_maxLongitude;
}

int Item::doCalculateNextIndex(bool randomMode, int currentIndex, int pages) const
{
  return (pages)? (currentIndex + ((randomMode)? (qrand() % 5) + 1 : 1)) % pages : 1;
}

QUrl Item::doPrepareInit(QHttpRequestHeader & /* header */)
{
  _photoId.clear();
  photoOwner.clear();
  photoTitle.clear();
  photoLocation.clear();
  _photoUrl.clear();
  _sourceUrl.clear();
  photoSize = QSize(-1,-1);

  return QUrl();
}

QUrl Item::doPrepareSearch(QHttpRequestHeader & /* header */)
{
  QUrl url("http://www.panoramio.com/map/get_panoramas.php");

  url.addQueryItem("order",searchOrderStrings[_searchOrder]);
  url.addQueryItem("set","public");
  url.addQueryItem("size",requestedSizeStrings[_requestedSize]);
  url.addQueryItem("from",QString::number(pageIndex()-1));
  url.addQueryItem("to",QString::number(pageIndex()));
  url.addQueryItem("minx",QString::number(_minLongitude));
  url.addQueryItem("miny",QString::number(_minLatitude));
  url.addQueryItem("maxx",QString::number(_maxLongitude));
  url.addQueryItem("maxy",QString::number(_maxLatitude));

  return url;
}

bool Item::doProcessSearchResult(const QByteArray &response, int &newPagesCount)
{
  QScriptValue value;
  QScriptEngine engine;

  value = engine.evaluate(response);
  newPagesCount = value.property("count").toInt32();

  if (value.property("photos").isArray())
  {
    QScriptValueIterator photo(value.property("photos"));

    while (photo.hasNext())
    {
      photo.next();

      _photoId = photo.value().property("photo_id").toString();
      _photoUrl = photo.value().property("photo_file_url").toString();
      _sourceUrl = photo.value().property("photo_url").toString();
      photoTitle = photo.value().property("photo_title").toString();
      photoOwner = photo.value().property("owner_name").toString();
      photoSize.setWidth(photo.value().property("width").toInt32());
      photoSize.setHeight(photo.value().property("height").toInt32());
      photoLocation = tr("lat:") + " " + photo.value().property("latitude").toString() + ", " +
                      tr("lon:") + " " + photo.value().property("longitude").toString();
    }
  }

  return ((_requestedSize == Original) && !_photoId.isEmpty() && (_lastPhotoId != _photoId));
}

QUrl Item::prepareSizeRequest(QHttpRequestHeader & /* header */)
{
  return QUrl("/");
}

QSize Item::doProcessSizeRequestResult(const QByteArray & /* response */)
{
  return photoSize;
}

QUrl Item::prepareDownload(QHttpRequestHeader &header)
{
  if (_requestedSize == Original)
  {
    header.setRequest("GET","/photos/original/" + _photoId + ".jpg");
    header.setValue("Host","static.panoramio.com");
    header.setValue("Referer","http://www.panoramio.com/photo/" + _photoId);

    return QUrl("http://static.panoramio.com/photos/original/" + _photoId + ".jpg");
  }
  else
    return _photoUrl;
}

QUrl Item::prepareInfoCollect(QHttpRequestHeader & /* header */)
{
  _lastPhotoId = _photoId;
  return QUrl("/");
}

PhotoInfo Item::doProcessInfoCollectResult(const QByteArray & /* response */)
{
  PhotoInfo info;

  info.title = photoTitle;
  info.owner = photoOwner;
  info.location = photoLocation;
  info.sourceUrl = _sourceUrl;
  info.searchString = QString("%1,%2,%3,%4").arg(_minLongitude).arg(_maxLatitude).arg(_maxLongitude).arg(_minLatitude);

  return info;
}

void Item::doLoadSettings(QSettings &settings)
{
  _minLongitude = settings.value(MIN_LONGITUDE,-180.0).toDouble();
  _minLatitude = settings.value(MIN_LATITUDE,-90.0).toDouble();
  _maxLongitude = settings.value(MAX_LONGITUDE,180.0).toDouble();
  _maxLatitude = settings.value(MAX_LATITUDE,90.0).toDouble();
  _searchOrder = static_cast<SearchOrder> (settings.value(SEARCH_ORDER,Popularity).toInt());
  _requestedSize = static_cast<RequestedSize> (settings.value(REQUESTED_SIZE,Original).toInt());
  _lastPhotoId = settings.value(LAST_PHOTO_ID,QString()).toString();
}

void Item::doSaveSettings(QSettings &settings) const
{
  settings.setValue(MIN_LONGITUDE,_minLongitude);
  settings.setValue(MIN_LATITUDE,_minLatitude);
  settings.setValue(MAX_LONGITUDE,_maxLongitude);
  settings.setValue(MAX_LATITUDE,_maxLatitude);
  settings.setValue(SEARCH_ORDER,_searchOrder);
  settings.setValue(REQUESTED_SIZE,_requestedSize);
}

void Item::doSaveState(QSettings &settings) const
{
  settings.setValue(LAST_PHOTO_ID,_lastPhotoId);
}

QVariant Item::data() const
{
  QString size, order;

  switch (_requestedSize)
  {
    case Original: size = tr("Original"); break;
    case Medium: size = tr("Medium"); break;
    case Small: size = tr("Small"); break;
    case Thumbnail: size = tr("Thumbnail"); break;
    case Square: size = tr("Square"); break;
    case MiniSquare: size = tr("Mini square"); break;
  }

  switch (_searchOrder)
  {
    case Popularity: order = tr("Popularity"); break;
    case UploadDate: order = tr("Upload date"); break;
  }

  return QString("<font size=\"+1\"><table cellspacing=\"3\"><tr><td><b>") + tr("Longitude:") +
         "</b></td><td align=\"right\">" + QString::number(minLongitude(),'f') + "</td>" +
         "<td align=\"center\">-></td><td align=\"right\">" + QString::number(maxLongitude(),'f') +
         "</td></tr><tr><td><b>" + tr("Latitude:") + "</b></td><td align=\"right\">" +
         QString::number(minLatitude(),'f') + "</td><td align=\"center\">-></td><td align=\"right\">" +
         QString::number(maxLatitude(),'f') + "</td></tr></table>" +
         "</font><i>&nbsp;&nbsp;" + tr("Size:") + "&nbsp;" + size + ",&nbsp;&nbsp;" +
         tr("Order:") + "&nbsp;" + order + "</i>";
}

bool Item::equalTo(Engine::Item *item) const
{
  Item *panoramioItem = qobject_cast<Item *> (item);

  return panoramioItem && (minLatitude() == panoramioItem->minLatitude()) &&
                          (maxLatitude() == panoramioItem->maxLatitude()) &&
                          (minLongitude() == panoramioItem->minLongitude()) &&
                          (maxLongitude() == panoramioItem->maxLongitude());
}

DialogWidget::DialogWidget(XtEngine::Core *core, QWidget *parent) : XtEngine::DialogWidget(core,parent)
{
  QVBoxLayout *mainLayout = new QVBoxLayout;
  QFormLayout *formLayout = new QFormLayout;
  QPushButton *pbMap = new QPushButton(tr("Select on map"),this);

  checkingValues = false;

  connect(pbMap,SIGNAL(clicked()),this,SLOT(launchMapViewer()));

  formLayout->addRow(new QLabel(tr("Longitude (min):"),this),
                     dsbMinLongitude = new QDoubleSpinBox(this));
  formLayout->addRow(new QLabel(tr("Longitude (max):"),this),
                     dsbMaxLongitude = new QDoubleSpinBox(this));
  formLayout->addRow(new QLabel(tr("Latitude (min):"),this),
                     dsbMinLatitude = new QDoubleSpinBox(this));
  formLayout->addRow(new QLabel(tr("Latitude (max):"),this),
                     dsbMaxLatitude = new QDoubleSpinBox(this));

  dsbMaxLongitude->setRange(-180.0,180.0);
  dsbMaxLongitude->setDecimals(6);
  connect(dsbMaxLongitude,SIGNAL(editingFinished()),this,SLOT(correctValues()));
  dsbMinLongitude->setRange(-180.0,180.0);
  dsbMinLongitude->setDecimals(6);
  connect(dsbMinLongitude,SIGNAL(editingFinished()),this,SLOT(correctValues()));
  dsbMaxLatitude->setRange(-90.0,90.0);
  dsbMaxLatitude->setDecimals(6);
  connect(dsbMaxLatitude,SIGNAL(editingFinished()),this,SLOT(correctValues()));
  dsbMinLatitude->setRange(-90.0,90.0);
  dsbMinLatitude->setDecimals(6);
  connect(dsbMinLatitude,SIGNAL(editingFinished()),this,SLOT(correctValues()));

  cbSearchOrder = new QComboBox(this);
  cbSearchOrder->setSizeAdjustPolicy(QComboBox::AdjustToContents);
  cbSearchOrder->addItem(tr("Popularity"),Popularity);
  cbSearchOrder->addItem(tr("Upload date"),UploadDate);
  cbSearchOrder->setCurrentIndex(0);

  cbRequestedSize = new QComboBox(this);
  cbRequestedSize->setSizeAdjustPolicy(QComboBox::AdjustToContents);
  cbRequestedSize->addItem(tr("Original"),Original);
  cbRequestedSize->addItem(tr("Medium"),Medium);
  cbRequestedSize->addItem(tr("Small"),Small);
  cbRequestedSize->addItem(tr("Thumbnail"),Thumbnail);
  cbRequestedSize->addItem(tr("Square"),Square);
  cbRequestedSize->addItem(tr("Mini square"),MiniSquare);
  cbRequestedSize->setCurrentIndex(0);

  formLayout->addRow(new QLabel(tr("Order:"),this),cbSearchOrder);
  formLayout->addRow(new QLabel(tr("Size:"),this),cbRequestedSize);

  mainLayout->addWidget(pbMap);
  mainLayout->addLayout(formLayout);

  setLayout(mainLayout);

  dsbMinLongitude->setFocus();
}

void DialogWidget::correctValues()
{
  if (!checkingValues)
  {
    checkingValues = true;

    dsbMaxLongitude->setValue(qMax(dsbMinLongitude->value(),dsbMaxLongitude->value()));
    dsbMinLongitude->setValue(qMin(dsbMinLongitude->value(),dsbMaxLongitude->value()));
    dsbMaxLatitude->setValue(qMax(dsbMinLatitude->value(),dsbMaxLatitude->value()));
    dsbMinLatitude->setValue(qMin(dsbMinLatitude->value(),dsbMaxLatitude->value()));

    checkingValues = false;
  }
}

XtEngine::Item *DialogWidget::returnItem()
{
  Item *item = qobject_cast<Item *> (core()->newItem());

  item->setMinLongitude(dsbMinLongitude->value());
  item->setMaxLongitude(dsbMaxLongitude->value());
  item->setMinLatitude(dsbMinLatitude->value());
  item->setMaxLatitude(dsbMaxLatitude->value());
  item->setSearchOrder(static_cast<SearchOrder>
                         (cbSearchOrder->itemData(cbSearchOrder->currentIndex()).toInt()));
  item->setRequestedSize(static_cast<RequestedSize>
                          (cbRequestedSize->itemData(cbRequestedSize->currentIndex()).toInt()));

  return item;
}

void DialogWidget::setupFromItem(XtEngine::Item *item)
{
  if (item)
  {
    Item *panoramioItem = qobject_cast<Item *> (item);

    setWindowTitle(tr("Edit Panoramio item"));

    checkingValues = true;
    dsbMinLatitude->setValue(panoramioItem->minLatitude());
    dsbMaxLatitude->setValue(panoramioItem->maxLatitude());
    dsbMinLongitude->setValue(panoramioItem->minLongitude());
    dsbMaxLongitude->setValue(panoramioItem->maxLongitude());
    checkingValues = false;

    cbSearchOrder->setCurrentIndex(cbSearchOrder->findData(panoramioItem->searchOrder()));
    cbRequestedSize->setCurrentIndex(cbRequestedSize->findData(panoramioItem->requestedSize()));
  }
  else
    setWindowTitle(tr("Add Panoramio item"));
}

bool DialogWidget::validateInput()
{
  if ((dsbMinLatitude->value() == dsbMaxLatitude->value()) ||
      (dsbMinLongitude->value() == dsbMaxLongitude->value()))
  {
    QMessageBox::critical(this,tr("Error"),tr("Coordinates must be different"));
    return false;
  }
  else
    return true;
}

void DialogWidget::launchMapViewer()
{
  Map::Viewer *viewer;

  if ((dsbMaxLongitude->value() > dsbMinLongitude->value()) &&
      (dsbMaxLatitude->value() > dsbMinLatitude->value()))
    viewer = new Map::Viewer(QRectF(QPointF(dsbMinLongitude->value(),
                                            dsbMaxLatitude->value()),
                                    QPointF(dsbMaxLongitude->value(),
                                            dsbMinLatitude->value())),this);
  else
    viewer = new Map::Viewer(this);

  if (viewer->exec() == QDialog::Accepted)
  {
    QRectF llRect = viewer->llRect();

    checkingValues = true;
    dsbMaxLongitude->setValue(llRect.right());
    dsbMinLongitude->setValue(llRect.left());
    dsbMaxLatitude->setValue(llRect.top());
    dsbMinLatitude->setValue(llRect.bottom());
    checkingValues = false;
  }

  delete viewer;
}
