/* dirfiltersets.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2007-2008 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "dirfiltersettings.hh"
#include "fileentry.hh"
#include "wconfig.h"
#include <algorithm>

DirFilterSettings::DirFilterSettings() : _serial_nr( 0 ),
                                         _show_hidden( true ),
                                         _ins( 0 ),
                                         _outs( 0 ),
                                         m_highlight_bookmark_prefix( true ),
                                         m_bookmark_filter( SHOW_ALL ),
                                         m_bookmark_label( "" )
{
}

DirFilterSettings::~DirFilterSettings()
{
}

bool DirFilterSettings::check( FileEntry &e )
{
    bool use = true;

    if ( _show_hidden == false &&
         e.isHidden == true &&
         strcmp( e.name, ".." ) != 0 ) {
        
        use = false;
    }

    if ( use == true ) {
        if ( _string_filter.get() != NULL &&
             strcmp( e.name, ".." ) != 0 ) {
            if ( _string_filter->match( e.name ) == false )
                use = false;
        }
    }

    if ( use == true ) {
        bool isIn;
        
        std::list<NM_Filter>::const_iterator it;
        
        if ( e.isDir() == true ) {
            // a dir is always visible
            use = true;
        } else {
            if ( _ins > 0 ) isIn = false;
            else isIn = true;
            
            for ( it = _filters.begin();
                  it != _filters.end();
                  it++ ) {
                if ( it->getCheck() > 0 ) {
                    if ( fnmatch( it->getPattern(), e.name, 0 ) == 0 ) {
                        if ( it->getCheck() == 1 ) isIn = true;
                        else isIn = false;
                        break;
                    }
                }
            }
            use = isIn;
        }
    }

    if ( use == true && m_bookmark_filter != SHOW_ALL ) {
        if ( m_bookmark_filter == SHOW_ONLY_BOOKMARKS ) {
            if ( e.getBookmarkMatch() == FileEntry::NOT_IN_BOOKMARKS ) {
                use = false;
            }
        } else if ( m_bookmark_filter == SHOW_ONLY_LABEL ) {
            if ( e.hasMatchingLabels() == true ) {
                try {
                    const std::list<std::string> &labels = e.getMatchingLabels();
                    
                    if ( std::find( labels.begin(),
                                    labels.end(),
                                    m_bookmark_label ) == labels.end() ) {
                        use = false;
                    }
                } catch ( int ) {
                    use = false;
                }
            } else {
                use = false;
            }
        }
    }

    if ( use == true ) {
        bool clear_override = true;

        if ( m_highlight_bookmark_prefix == true || e.getBookmarkMatch() == FileEntry::BOOKMARK ) {
            if ( e.hasMatchingLabels() == true ) {
                std::map<std::string, WConfig::ColorDef::label_colors_t>::const_iterator label_it;
                
                try {
                    const std::list<std::string> &matching_labels = e.getMatchingLabels();
                    const std::map<std::string, WConfig::ColorDef::label_colors_t> &label_colors = wconfig->getColorDefs().getLabelColors();
                    
                    if ( matching_labels.empty() == false ) {
                        if ( m_bookmark_filter != SHOW_ONLY_LABEL ) {
                            label_it = label_colors.find( matching_labels.front() );
                        } else {
                            label_it = label_colors.find( m_bookmark_label );
                        }
                        if ( label_it != label_colors.end() ) {
                            WConfig::ColorDef::label_colors_t col = label_it->second;
                            e.setOverrideColor( FileEntryCustomColor::NORMAL,
                                                col.normal_fg,
                                                col.normal_bg );
                            e.setOverrideColor( FileEntryCustomColor::ACTIVE,
                                                col.active_fg,
                                                col.active_bg );
                            clear_override = false;
                        }
                    }
                } catch ( int ) {
                }
            }
        }
        if ( clear_override == true ) {
            e.clearOverrideColor();
        }
    }

    e.use = use;
    return use;
}

void DirFilterSettings::setShowHidden( bool nv )
{
    _show_hidden = nv;
    _serial_nr++;
}

bool DirFilterSettings::getShowHidden() const
{
    return _show_hidden;
}

int DirFilterSettings::getSerialNr() const
{
    return _serial_nr;
}

void DirFilterSettings::setFilters( const std::list<NM_Filter> &filters )
{
    _filters = filters;
    calcInAndOuts();
    _serial_nr++;
}

std::list<NM_Filter> DirFilterSettings::getFilters()
{
    return _filters;
}

bool DirFilterSettings::filterActive() const
{
    bool res = false;

    std::list<NM_Filter>::const_iterator it;
    
    for ( it = _filters.begin();
          it != _filters.end();
          it++ ) {
        if ( it->getCheck() != 0 ) {
            res = true;
            break;
        }
    }

    return res;
}

void DirFilterSettings::unsetAllFilters()
{
    std::list<NM_Filter>::iterator it;
    
    for ( it = _filters.begin();
          it != _filters.end();
          it++ ) {
        it->setCheck( NM_Filter::INACTIVE );
    }
    
    calcInAndOuts();
    _serial_nr++;
}

void DirFilterSettings::setFilter( const char *filter, NM_Filter::check_t mode )
{
    std::list<NM_Filter>::iterator it;
    
    for ( it = _filters.begin();
          it != _filters.end();
          it++ ) {
        if ( strcmp( it->getPattern(), filter ) == 0 )
            break;
    }

    if ( it != _filters.end() ) {
        it->setCheck( mode );
    } else {
        NM_Filter fi;
        fi.setCheck( mode );
        fi.setPattern( filter );
        _filters.push_back( fi );
    }
    
    calcInAndOuts();
    _serial_nr++;
}

void DirFilterSettings::calcInAndOuts()
{
    std::list<NM_Filter>::const_iterator it;
        
    _ins = 0;
    _outs = 0;
    
    for ( it = _filters.begin();
          it != _filters.end();
          it++ ) {
        if ( it->getCheck() == 1 ) _ins++;
        if ( it->getCheck() == 2 ) _outs++;
    }
}

void DirFilterSettings::setStringFilter( std::auto_ptr<StringMatcher> filter )
{
    _string_filter = filter;

    _serial_nr++;
}

void DirFilterSettings::bookmarksChanged()
{
    _serial_nr++;
}

void DirFilterSettings::setHighlightBookmarkPrefix( bool nv )
{
    m_highlight_bookmark_prefix = nv;

    _serial_nr++;
}

bool DirFilterSettings::getHighlightBookmarkPrefix() const
{
    return m_highlight_bookmark_prefix;
}

void DirFilterSettings::setBookmarkFilter( bookmark_filter_t v )
{
    m_bookmark_filter = v;

    _serial_nr++;
}

DirFilterSettings::bookmark_filter_t DirFilterSettings::getBookmarkFilter() const
{
    return m_bookmark_filter;
}

void DirFilterSettings::setSpecificBookmarkLabel( const std::string &l )
{
    m_bookmark_label = l;

    _serial_nr++;
}

const std::string &DirFilterSettings::getSpecificBookmarkLabel() const
{
    return m_bookmark_label;
}

NM_Filter::check_t DirFilterSettings::checkFilter( const char *filter ) throw( int )
{
    std::list<NM_Filter>::iterator it;
    
    for ( it = _filters.begin();
          it != _filters.end();
          ++it ) {
        if ( strcmp( it->getPattern(), filter ) == 0 ) {
            break;
        }
    }

    if ( it == _filters.end() ) {
        throw 1;
    }

    return it->getCheck();
}
