/**********************************************************************
zyGrib: meteorological GRIB file viewer
Copyright (C) 2008 - Jacques Zaninetti - http://www.zygrib.org

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
***********************************************************************/

#include <QMessageBox>
#include <cmath>
#include <cassert>

#include "DialogLoadGRIB.h"
#include "DialogServerStatus.h"
#include "Util.h"


//-------------------------------------------------------------------------------
DialogLoadGRIB::DialogLoadGRIB() : QDialog()
{
    loadgrib = new FileLoaderGRIB();
    assert(loadgrib);

    oldcursor = cursor();
    
    setWindowTitle(tr("Téléchargement - GRIB"));
    loadInProgress = false;
    QFrame * frameButtonsZone = createFrameButtonsZone(this);

    rain  = true;
    cloud = true;
    pressure = true;
    wind  = true;
    temp  = true;
    humid = true;
    isotherm0 = true;
    tempPot = true;
    tempMin = true;
    tempMax = true;
    snowDepth = true;
    snowCateg = true;
    CAPEsfc = true;

    QGridLayout  *lay = new QGridLayout(this);
    assert(lay);
    lay->addWidget( frameButtonsZone,1,0, Qt::AlignLeft);

    connect(btCancel, SIGNAL(clicked()), this, SLOT(slotBtCancel()));
    connect(btOK, SIGNAL(clicked()), this, SLOT(slotBtOK()));
    connect(btServerStatus, SIGNAL(clicked()), this, SLOT(slotBtServerStatus()));

    connect(loadgrib, SIGNAL(signalGribDataReceived(QByteArray *, QString)),
            this,  SLOT(slotGribDataReceived(QByteArray *, QString)));
    connect(loadgrib, SIGNAL(signalGribReadProgress(int, int, int)),
            this,  SLOT(slotGribReadProgress(int, int, int)));
    connect(loadgrib, SIGNAL(signalGribLoadError(QString)),
            this,  SLOT(slotGribFileError(QString)));
    connect(loadgrib, SIGNAL(signalGribSendMessage(QString)),
            this,  SLOT(slotGribMessage(QString)));
    connect(loadgrib, SIGNAL(signalGribStartLoadData()),
            this,  SLOT(slotGribStartLoadData()));
            
    //------------------------------------------------------
    connect(sbNorth, SIGNAL(valueChanged(double)),
            this,  SLOT(slotParameterUpdated()));
    connect(sbSouth, SIGNAL(valueChanged(double)),
            this,  SLOT(slotParameterUpdated()));
    connect(sbWest, SIGNAL(valueChanged(double)),
            this,  SLOT(slotParameterUpdated()));
    connect(sbEast, SIGNAL(valueChanged(double)),
            this,  SLOT(slotParameterUpdated()));
            
    connect(cbResolution, SIGNAL(activated(int)),
            this,  SLOT(slotParameterUpdated()));
    connect(cbInterval, SIGNAL(activated(int)),
            this,  SLOT(slotParameterUpdated()));
    connect(cbDays, SIGNAL(activated(int)),
            this,  SLOT(slotParameterUpdated()));
    
    connect(chkWind, SIGNAL(stateChanged(int)), 	this,  SLOT(slotParameterUpdated()));
    connect(chkPressure, SIGNAL(stateChanged(int)), this,  SLOT(slotParameterUpdated()));
    connect(chkRain, SIGNAL(stateChanged(int)), 	this,  SLOT(slotParameterUpdated()));
    connect(chkCloud, SIGNAL(stateChanged(int)), 	this,  SLOT(slotParameterUpdated()));
    connect(chkTemp, SIGNAL(stateChanged(int)), 	this,  SLOT(slotParameterUpdated()));
    connect(chkHumid, SIGNAL(stateChanged(int)), 	this,  SLOT(slotParameterUpdated()));
    connect(chkIsotherm0, SIGNAL(stateChanged(int)), this,  SLOT(slotParameterUpdated()));
                
    connect(chkTempPot, SIGNAL(stateChanged(int)), 	this,  SLOT(slotParameterUpdated()));
    connect(chkTempMin, SIGNAL(stateChanged(int)), 	this,  SLOT(slotParameterUpdated()));
    connect(chkTempMax, SIGNAL(stateChanged(int)), 	this,  SLOT(slotParameterUpdated()));
    connect(chkSnowCateg, SIGNAL(stateChanged(int)), 	this,  SLOT(slotParameterUpdated()));
    connect(chkSnowDepth, SIGNAL(stateChanged(int)), 	this,  SLOT(slotParameterUpdated()));
    connect(chkCAPEsfc, SIGNAL(stateChanged(int)), 	this,  SLOT(slotParameterUpdated()));
}

//-------------------------------------------------------------------------------
DialogLoadGRIB::~DialogLoadGRIB()
{
    if (loadgrib != NULL)
        delete loadgrib;
}

//----------------------------------------------------
void DialogLoadGRIB::slotGribMessage(QString msg)
{
    labelMsg->setText(msg);
}

//----------------------------------------------------
void DialogLoadGRIB::slotGribDataReceived(QByteArray *content, QString fileName)
{
	setCursor(oldcursor);

	QString path = Util::getSetting("gribFilePath", "").toString();
	if (path == "")
		path = "./";
	else
		path += "/";
	// version >= 2.2.0 : file is bziped
	fileName = fileName + ".bz2";
    fileName = QFileDialog::getSaveFileName(this,
                 tr("Sauvegarde du fichier GRIB"), path+fileName, "");
    if (fileName != "")
    {
        QFile saveFile (fileName);
        bool ok;
        qint64 nb = 0;
        ok = saveFile.open(QIODevice::WriteOnly);
        if (ok) {
            nb = saveFile.write(*content);
            if (nb > 0) {
            	QFileInfo info(saveFile);
				Util::setSetting("gribFilePath", info.absolutePath() );
                saveFile.close();
            }
        }
        if (ok && nb>0) {
            loadInProgress = false;
            btCancel->setText(tr("Annuler"));
            btOK->setEnabled(true);
            accept();
            emit signalMeteoFileReceived(fileName);
        }
        else {
            QMessageBox::critical (this,
                    tr("Erreur"),
                    tr("Ecriture du fichier impossible."));
        }
    }
    else {
        QMessageBox::critical (this,
                tr("Erreur"),
                tr("Opération abandonnée."));
        loadInProgress = false;
        btCancel->setText(tr("Annuler"));
        btOK->setEnabled(true);
        slotGribReadProgress(1,0,100);
    }
}

//----------------------------------------------------
void DialogLoadGRIB::slotGribFileError(QString error)
{
	setCursor(oldcursor);
    if (! loadInProgress)
        return;
    
    QString s;
    QMessageBox::critical (this,
                    tr("Erreur"),
                    tr("Erreur : ") + error );
    
    loadInProgress = false;
    btCancel->setText(tr("Annuler"));
    btOK->setEnabled(true);
    labelMsg->setText("");
}

//----------------------------------------------------
void DialogLoadGRIB::slotGribStartLoadData()
{
    timeLoad.start();
}

//----------------------------------------------------
void DialogLoadGRIB::slotGribReadProgress(int step, int done, int total)
{
    if (step < 2) {
        progressBar->setRange(0,1000);
        progressBar->setValue(step);
    }
    else {
        progressBar->setRange(0,total);
        progressBar->setValue(done);
/****
        // temps estimé
        int elapsed = timeLoad.elapsed();
        QTime duree(0,0);
        duree = duree.addMSecs(elapsed);
        QString eta = duree.toString("HH:mm:ss");
     ****/
        
        slotGribMessage(tr("Taille : %1 ko    Reçus : %2 ko ")
                .arg( total/1024 )
                .arg( done/1024 )
        );
    }
}

//-------------------------------------------------------------------------------
void DialogLoadGRIB::updateParameters()
{
    float tmp, xm, ym;
    
    xmin = sbWest->cleanText().toFloat();
    xmax = sbEast->cleanText().toFloat();
    ymin = sbNorth->cleanText().toFloat();
    ymax = sbSouth->cleanText().toFloat();

    resolution = cbResolution->currentText().toFloat();
    interval   = cbInterval->currentText().toInt();
    days       = cbDays->currentText().toInt();

    if (xmin > xmax) {
        tmp = xmin;   xmin = xmax;   xmax = tmp;
    }
    if (ymin < ymax) {    // échelle Y inversée (90=nord)
        tmp = ymin;   ymin = ymax;   ymax = tmp;
    }
    // trop grand ?
    if (fabs(xmax-xmin) >=360)
        xmax = xmin+359.9;
    if (fabs(ymin-ymax) >=180)
        ymin = ymax+179.9;
    
    // trop petit ?
    if (fabs(xmax-xmin) < 2*resolution) {
        xm = (xmin+xmax)/2;
        xmin = xm - 2*resolution;
        xmax = xm + 2*resolution;
    }
    if (fabs(ymin-ymax) < 2*resolution) {
        ym = (ymin+ymax)/2;
        ymin = ym + 2*resolution;
        ymax = ym - 2*resolution;
    }

    wind     = chkWind->isChecked();
    pressure = chkPressure->isChecked();
    rain     = chkRain->isChecked();
    cloud    = chkCloud->isChecked();
    temp     = chkTemp->isChecked();
    humid    = chkHumid->isChecked();
    isotherm0    = chkIsotherm0->isChecked();
	
    tempPot     = chkTempPot->isChecked();
    tempMin     = chkTempMin->isChecked();
    tempMax     = chkTempMax->isChecked();
    snowDepth   = chkSnowDepth->isChecked();
    snowCateg   = chkSnowCateg->isChecked();
    frzRainCateg = chkFrzRainCateg->isChecked();
    CAPEsfc      = chkCAPEsfc->isChecked();
	
	Util::setSetting("downloadIndResolution", cbResolution->currentIndex());
	Util::setSetting("downloadIndInterval",  cbInterval->currentIndex());
	Util::setSetting("downloadIndNbDays",  cbDays->currentIndex());
	
	Util::setSetting("downloadWind",  wind);
	Util::setSetting("downloadPressure", pressure);
	Util::setSetting("downloadRain",  rain);
	Util::setSetting("downloadCloud", cloud);
	Util::setSetting("downloadTemp",  temp);
	Util::setSetting("downloadHumid", humid);
	Util::setSetting("downloadIsotherm0", isotherm0);
	
	Util::setSetting("downloadTempPot",  tempPot);
	Util::setSetting("downloadTempMin",  tempMin);
	Util::setSetting("downloadTempMax",  tempMax);
	Util::setSetting("downloadSnowDepth", snowDepth);
	Util::setSetting("downloadSnowCateg", snowCateg);
	Util::setSetting("downloadFrzRainCateg", frzRainCateg);
	Util::setSetting("downloadCAPEsfc", CAPEsfc);
}

//-------------------------------------------------------------------------------
void DialogLoadGRIB::slotParameterUpdated()
{
    updateParameters();
    
    int npts = (int) (  ceil(fabs(xmax-xmin)/resolution)
                       * ceil(fabs(ymax-ymin)/resolution) );
    
    // Nombre de GribRecords
    int nbrec = (int) days*24/interval +1;
    int nbPress = pressure ?  nbrec   : 0;
    int nbWind  = wind     ?  2*nbrec : 0;
    int nbRain  = rain     ?  nbrec-1 : 0;
    int nbCloud = cloud    ?  nbrec-1 : 0;
    int nbTemp  = temp     ?  nbrec   : 0;
    int nbHumid = humid    ?  nbrec   : 0;
    int nbIsotherm0 = isotherm0    ?  nbrec   : 0;
    
    int nbTempPot  = tempPot ?  nbrec   : 0;
    int nbTempMin  = tempMin ?  nbrec-1  : 0;
    int nbTempMax  = tempMax ?  nbrec-1  : 0;
    int nbSnowDepth  = snowDepth ?  nbrec  : 0;
    int nbSnowCateg  = snowCateg ?  nbrec-1  : 0;
    int nbFrzRainCateg = frzRainCateg ?  nbrec-1  : 0;
    int nbCAPEsfc  = CAPEsfc ?  nbrec : 0;
    
    
//    int estime = (nbWind+nbPress+nbRain+nbCloud+nbTemp) * (nbx*nby*2+84);
    int head = 84;
    int estime = 0;
    int nbits;
    
    nbits = 11;
    estime += nbTemp*(head+(nbits*npts)/8+2 );
    estime += nbTempPot*(head+(nbits*npts)/8+2 );
    estime += nbTempMin*(head+(nbits*npts)/8+2 );
    estime += nbTempMax*(head+(nbits*npts)/8+2 );
    estime += nbSnowDepth*(head+(nbits*npts)/8+2 );
    nbits = 12;
    estime += nbRain*(head+(nbits*npts)/8+2 );
    nbits = 13;
    estime += nbWind*(head+(nbits*npts)/8+2 );
    nbits = 16;
    estime += nbPress*(head+(nbits*npts)/8+2 );
    nbits = 8;
    estime += nbCloud*(head+(nbits*npts)/8+2 );
    nbits = 7;
    estime += nbSnowCateg*(head+(nbits*npts)/8+2 );
    estime += nbFrzRainCateg*(head+(nbits*npts)/8+2 );
    nbits = 10;
    estime += nbHumid*(head+(nbits*npts)/8+2 );
    estime += nbCAPEsfc*(head+(nbits*npts)/8+2 );
    nbits = 16;
    estime += nbIsotherm0*(head+(nbits*npts)/8+2 );

    estime = estime/1024;
    
    slotGribMessage(tr("Taille estimée : environ %1 ko").arg(estime) );
    
    if (estime == 0)
        btOK->setEnabled(false);
    else
        btOK->setEnabled(true);

}

//-------------------------------------------------------------------------------
void DialogLoadGRIB::slotBtOK()
{
    btCancel->setText(tr("Stop"));

    loadInProgress = true;
    btOK->setEnabled(false);
    
    setCursor(Qt::WaitCursor);
    loadgrib->getGribFile(
					xmin,ymin, xmax,ymax,
					resolution, interval, days,
					wind, pressure, rain, cloud, temp, humid, isotherm0,
					tempPot, tempMin, tempMax, snowDepth, snowCateg, frzRainCateg,
					CAPEsfc
				);
}
//-------------------------------------------------------------------------------
void DialogLoadGRIB::slotBtServerStatus()
{
    DialogServerStatus dial;
    dial.exec();
}

//-------------------------------------------------------------------------------
void DialogLoadGRIB::slotBtCancel()
{
	setCursor(oldcursor);

    if (loadInProgress)
    {
        loadInProgress = false;
        loadgrib->stop();
        btCancel->setText(tr("Annuler"));
        progressBar->setRange(0,100);
        progressBar->setValue(0);
        slotParameterUpdated();
    }
    else {
        reject();
    }
}

//-------------------------------------------------------------------------------
void DialogLoadGRIB::setZone(float x0, float y0, float x1, float y1)
{
    sbNorth->setValue(y0);
    sbSouth->setValue(y1);
    sbWest->setValue(x0);
    sbEast->setValue(x1);
    progressBar->setRange(0,100);
    progressBar->setValue(0);
    slotParameterUpdated();
}

//-------------------------------------------------------------------------------
QFrame *DialogLoadGRIB::createFrameButtonsZone(QWidget *parent)
{
    QFrame * frm = new QFrame(parent);
    assert(frm);
    QVBoxLayout  *lay = new QVBoxLayout(frm);
    assert(lay);
    int ind, lig,col;
    
    QFrame *ftmp, *frline;

	//------------------------------------------------
	// Geographic area
	//------------------------------------------------
    int sizemin = 0;
    sbNorth = new QDoubleSpinBox(this);
    assert(sbNorth);
    sbNorth->setDecimals(0);
    sbNorth->setMinimum(-90);
    sbNorth->setMaximum(90);
    sbNorth->setSuffix(tr(" °N"));
    
    sbSouth = new QDoubleSpinBox(this);
    assert(sbSouth);
    sbSouth->setDecimals(0);
    sbSouth->setMinimum(-90);
    sbSouth->setMaximum(90);
    sbSouth->setSuffix(tr(" °N"));
    
    sbWest = new QDoubleSpinBox(this);
    assert(sbWest);
    sbWest->setDecimals(0);
    sbWest->setMinimum(-360);
    sbWest->setMaximum(360);
    sbWest->setSuffix(tr(" °E"));
    
    sbEast = new QDoubleSpinBox(this);
    assert(sbEast);
    sbEast->setDecimals(0);
    sbEast->setMinimum(-360);
    sbEast->setMaximum(360);
    sbEast->setSuffix(tr(" °E"));

	//------------------------------------------------
	// Résolution, intervalle, durée
	//------------------------------------------------
    cbResolution = new QComboBox(this);
    assert(cbResolution);
    cbResolution->addItems(QStringList()<< "0.5" << "1" << "2");
    cbResolution->setMinimumWidth (sizemin);
	ind = Util::getSetting("downloadIndResolution", 1).toInt();
	ind = Util::inRange(ind, 0, cbResolution->count()-1);
    cbResolution->setCurrentIndex(ind);
    
    cbInterval = new QComboBox(this);
    assert(cbInterval);
    cbInterval->addItems(QStringList()<< "3" << "6" << "12" << "24");
    cbInterval->setMinimumWidth (sizemin);
	ind = Util::getSetting("downloadIndInterval", 1).toInt();
	ind = Util::inRange(ind, 0, cbInterval->count()-1);
    cbInterval->setCurrentIndex(ind);
    
    cbDays = new QComboBox(this);
    assert(cbDays);
    cbDays->addItems(QStringList()<< "1"<<"2"<<"3"<<"4"<<"5"<<"6"<<"7");
    cbDays->setMinimumWidth (sizemin);
	ind = Util::getSetting("downloadIndNbDays", 4).toInt();	
	ind = Util::inRange(ind, 0, cbDays->count()-1);
    cbDays->setCurrentIndex(ind);

	//------------------------------------------------
	// Choix des données météo
	//------------------------------------------------
    chkWind     = new QCheckBox(tr("Vent (10 m)"));
    assert(chkWind);
    chkPressure = new QCheckBox(tr("Pression au niveau de la mer"));
    assert(chkPressure);
    chkRain     = new QCheckBox(tr("Cumul de précipitations"));
    assert(chkRain);
    chkCloud     = new QCheckBox(tr("Nébulosité"));
    assert(chkCloud);
    chkTemp     = new QCheckBox(tr("Température (2 m)"));
    assert(chkTemp);
    chkHumid    = new QCheckBox(tr("Humidité relative (2 m)"));
    assert(chkHumid);
    chkIsotherm0    = new QCheckBox(tr("Isotherme 0°C"));
    assert(chkIsotherm0);
    
    chkTempPot     = new QCheckBox(tr("Température potentielle (sigma 995)"));
    assert(chkTempPot);
    chkTempMin     = new QCheckBox(tr("Température min (2 m)"));
    assert(chkTempMin);
    chkTempMax     = new QCheckBox(tr("Température max (2 m)"));
    assert(chkTempMax);
    chkSnowCateg     = new QCheckBox(tr("Neige (chute possible)"));
    assert(chkSnowCateg);
    chkFrzRainCateg     = new QCheckBox(tr("Pluie verglaçante (chute possible)"));
    assert(chkFrzRainCateg);
    chkSnowDepth     = new QCheckBox(tr("Neige (épaisseur)"));
    assert(chkSnowDepth);
    chkCAPEsfc     = new QCheckBox(tr("CAPE (surface)"));
    assert(chkCAPEsfc);

    chkAltitudeData  = new QCheckBox(
    	tr("Atmosphère (850 mb, 700 mb, 500 mb) : altitude, température, humidité spécifique.")
    	+"\n"
    	+tr("Attention : ces données augmentent sensiblement la taille du fichier GRIB."));
    assert(chkAltitudeData);
    
    chkWind->setChecked    (Util::getSetting("downloadWind", true).toBool());
    chkPressure->setChecked(Util::getSetting("downloadPressure", true).toBool());
    chkRain->setChecked    (Util::getSetting("downloadRain", true).toBool());
    chkCloud->setChecked   (Util::getSetting("downloadCloud", true).toBool());
    chkTemp->setChecked    (Util::getSetting("downloadTemp", true).toBool());
    chkHumid->setChecked   (Util::getSetting("downloadHumid", true).toBool());
    chkIsotherm0->setChecked  (Util::getSetting("downloadIsotherm0", true).toBool());
    
    chkTempPot->setChecked    (Util::getSetting("downloadTempPot", true).toBool());
    chkTempMin->setChecked    (Util::getSetting("downloadTempMin", true).toBool());
    chkTempMax->setChecked    (Util::getSetting("downloadTempMax", true).toBool());
    chkSnowDepth->setChecked  (Util::getSetting("downloadSnowDepth", true).toBool());
    chkSnowCateg->setChecked  (Util::getSetting("downloadSnowCateg", true).toBool());
    chkCAPEsfc->setChecked  (Util::getSetting("downloadCAPEsfc", false).toBool());
    chkFrzRainCateg->setChecked  (Util::getSetting("downloadFrzRainCateg", true).toBool());

    chkAltitudeData->setChecked  (Util::getSetting("downloadAltitudeData", false).toBool());

	//----------------------------------------------------------------
    btOK     = new QPushButton(tr("Télécharger le fichier GRIB"), this);
    assert(btOK);
    btCancel = new QPushButton(tr("Annuler"), this);
    assert(btCancel);
    btServerStatus = new QPushButton(tr("Statut du serveur"), this);
    assert(btServerStatus);

    progressBar = new QProgressBar();
    assert(progressBar);

    QLayout  *tlay;
    QGridLayout  *tgrid;
	//------------------------------------------------
	// Disposition des widgets
	//------------------------------------------------
	
	ftmp = new QFrame(this);
    tgrid = new QGridLayout(ftmp);
    assert(tgrid);
		tgrid->addWidget( new QLabel(tr("Latitude min :")), 0, 0, Qt::AlignRight);
		tgrid->addWidget( sbNorth, 0, 1);
		tgrid->addWidget( new QLabel(tr("Latitude max :")), 0, 2, Qt::AlignRight);
		tgrid->addWidget( sbSouth, 0, 3);
		
		tgrid->addWidget( new QLabel(tr("Longitude min :")), 1, 0, Qt::AlignRight);
		tgrid->addWidget( sbWest, 1, 1);
		tgrid->addWidget( new QLabel(tr("Longitude max :")), 1, 2, Qt::AlignRight);
		tgrid->addWidget( sbEast, 1, 3);

    lay->addWidget( ftmp);
    
    //-------------------------
    ftmp = new QFrame(this); ftmp->setFrameShape(QFrame::HLine); lay->addWidget( ftmp);
    //-------------------------
	
	ftmp = new QFrame(this);
    tlay = new QHBoxLayout(ftmp);
    assert(tlay);
		
		tlay->addWidget( new QLabel(tr("Résolution :")));
		tlay->addWidget( cbResolution);
		tlay->addWidget( new QLabel(tr(" °")));
		//-------------------------
		frline=new QFrame(this); frline->setFrameShape(QFrame::VLine);tlay->addWidget(frline);
		//-------------------------
		tlay->addWidget( new QLabel(tr("Intervalle :")));
		tlay->addWidget( cbInterval);
		tlay->addWidget( new QLabel(tr(" heures")));
		//-------------------------
		frline=new QFrame(this); frline->setFrameShape(QFrame::VLine);tlay->addWidget(frline);
		//-------------------------
		tlay->addWidget( new QLabel(tr("Durée :")));
		tlay->addWidget( cbDays);
		tlay->addWidget( new QLabel(tr(" jours")));
    lay->addWidget( ftmp);
    
    //-------------------------
    ftmp = new QFrame(this); ftmp->setFrameShape(QFrame::HLine); lay->addWidget( ftmp);
    //-------------------------
	ftmp = new QFrame(this);
    tgrid = new QGridLayout(ftmp);
    assert(tgrid);
    	//-----------------------------
    	// Colonne 1
    	//-----------------------------
    	col = 0;
    	lig = 0;
		tgrid->addWidget( chkWind ,       lig++, col, Qt::AlignLeft);
		tgrid->addWidget( chkPressure ,   lig++, col, Qt::AlignLeft);
		tgrid->addWidget( chkTemp ,      lig++, col, Qt::AlignLeft);
		tgrid->addWidget( chkTempMin ,   lig++, col, Qt::AlignLeft);
		tgrid->addWidget( chkTempMax ,   lig++, col, Qt::AlignLeft);
		tgrid->addWidget( chkTempPot , lig++, col, Qt::AlignLeft);
		tgrid->addWidget( chkFrzRainCateg , lig++, col, Qt::AlignLeft);
    	//-----------------------------
    	// Colonne 2
    	//-----------------------------
    	col = 1;
    	lig = 0;
		tgrid->addWidget( chkIsotherm0 , lig++, col, Qt::AlignLeft);
		tgrid->addWidget( chkCloud ,   lig++, col, Qt::AlignLeft);
		tgrid->addWidget( chkHumid ,   lig++, col, Qt::AlignLeft);
		tgrid->addWidget( chkRain ,    lig++, col, Qt::AlignLeft);
		tgrid->addWidget( chkSnowCateg ,  lig++, col, Qt::AlignLeft);
		tgrid->addWidget( chkSnowDepth ,  lig++, col, Qt::AlignLeft);
		tgrid->addWidget( chkCAPEsfc ,  lig++, col, Qt::AlignLeft);
		
		//tgrid->addWidget( chkAltitudeData , lig++, 0,1,-1, Qt::AlignLeft);
    lay->addWidget( ftmp);
    
    //-------------------------
    ftmp = new QFrame(this); ftmp->setFrameShape(QFrame::HLine); lay->addWidget( ftmp);
    //-------------------------
    lay->addWidget( progressBar );
    //-------------------------
    labelMsg = new QLabel();
    lay->addWidget( labelMsg );
    lay->addWidget(new QLabel(tr("La taille des fichiers est limitée à 20000 ko.")));
    
    //-------------------------
    ftmp = new QFrame(this); ftmp->setFrameShape(QFrame::HLine); lay->addWidget( ftmp);
    //-------------------------
	ftmp = new QFrame(this);
    tlay = new QHBoxLayout(ftmp);
    assert(tlay);
		tlay->addWidget( btOK );
		tlay->addWidget( btServerStatus );
		tlay->addWidget( btCancel );
    lay->addWidget( ftmp);

    return frm;
}











