/**********************************************************************
zyGrib: meteorological GRIB file viewer
Copyright (C) 2008-2010 - Jacques Zaninetti - http://www.zygrib.org

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
***********************************************************************/

#include "DataQString.h"

//-------------------------------------------------
QString AltitudeStr::toString (const Altitude &alt)
{
	QString res;
	switch (alt.levelType) {
		case  LV_GND_SURF:
			res = tr("surface");
			break;
		case  LV_ISOTHERM0:
			res = tr("isotherm 0°C");
			break;
		case  LV_ISOBARIC:
			res = tr("%1 hPa").arg(alt.levelValue);
			break;
		case  LV_MSL :
			res = tr("sea level");
			break;
		case  LV_ABOV_GND:
			res = tr("%1 m above ground").arg(alt.levelValue);
			break;
		case  LV_SIGMA:
			res = tr("sigma %1").arg((int)(alt.levelValue/10));
			break;
		case  LV_ATMOS_ALL:
			res = tr("atmosphere");
			break;
		default:
			res =  "";
	}
	return res;
}
//-------------------------------------------------
QString AltitudeStr::toStringShort (const Altitude &alt)
{
	QString res;
	switch (alt.levelType) {
		case  LV_GND_SURF:
			res = tr("sfc");
			break;
		case  LV_ISOTHERM0:
			res = tr("iso0°C");
			break;
		case  LV_ISOBARIC:
			res = tr("%1hpa").arg(alt.levelValue);
			break;
		case  LV_MSL :
			res = tr("MSL");
			break;
		case  LV_ABOV_GND:
			res = tr("%1 m").arg(alt.levelValue);
			break;
		case  LV_SIGMA:
			res = tr("s%1").arg((int)(alt.levelValue/10));
			break;
		case  LV_ATMOS_ALL:
			res = tr("atm");
			break;
		default:
			res =  "";
	}
	return res;
}
//-------------------------------------------------
QString DataCodeStr::toString_name (const DataCode &dtc)
{
	switch (dtc.dataType) {
		case GRB_PRESSURE     : return tr("Pressure");
		case GRB_GEOPOT_HGT   : return tr("Geopotential");
		case GRB_TEMP         : return tr("Temperature");
		case GRB_TEMP_POT     : return tr("Potential temperature");
		case GRB_TMAX         : return tr("Temperature max");
		case GRB_TMIN         : return tr("Temperature min");
		case GRB_DEWPOINT     : return tr("Dew point");
		case GRB_WIND_VX      : return tr("Wind (Vx)");
		case GRB_WIND_VY      : return tr("Wind (Vy)");
		case GRB_HUMID_SPEC   : return tr("Specific humidity");
		case GRB_HUMID_REL    : return tr("Relative humidity");
		case GRB_PRECIP_RATE  : return tr("Precipitation rate");
		case GRB_PRECIP_TOT   : return tr("Total precipitation");
		case GRB_SNOW_DEPTH   : return tr("Snow (depth)");
		case GRB_CLOUD_TOT    : return tr("Cloud cover");
		case GRB_FRZRAIN_CATEG: return tr("Frozen rain possible");
		case GRB_SNOW_CATEG   : return tr("Snowfall possible");
		case GRB_CAPE 		  : return tr("CAPE");

		case GRB_WIND_XY2D    : return tr("Wind");
		case GRB_DIFF_TEMPDEW : return tr("Gap temperature-dew point");
		case GRB_THETA_E      : return tr("Theta-e");
			
		default:
			return  "";
	}
}
//-------------------------------------------------
QString DataCodeStr::toString_level (const DataCode &dtc)
{
	return AltitudeStr::toString (dtc.getAltitude());
}
//-------------------------------------------------
QString DataCodeStr::toString (const DataCode &dtc)
{
	QString name = toString_name (dtc);
	QString level = toString_level (dtc);
	if (level != "")
			return name+" (" +level+ ")";
	else
			return name;
}
//-------------------------------------------------
QString DataCodeStr::toString_levelShort (const DataCode &dtc)
{
	QString name = toString_name (dtc);
	QString level = AltitudeStr::toStringShort (dtc.getAltitude());
	if (level != "")
			return name+" (" +level+ ")";
	else
			return name;
}

//-------------------------------------------------
QString AltitudeStr::serialize (const Altitude &alt)
{
	return QString("%1;%2").arg(alt.levelType).arg(alt.levelValue);
}
//-------------------------------------------------
Altitude AltitudeStr::unserialize (const QString &stringCode)
{
	Altitude alt;
	QStringList list = stringCode.split(";");
	if (list.size() == 2) {
		bool ok;
		alt.levelType = list.at(0).toInt (&ok);
		if (!ok) alt.levelType = LV_TYPE_NOT_DEFINED;
		alt.levelValue = list.at(1).toInt (&ok);
		if (!ok) alt.levelType = LV_TYPE_NOT_DEFINED;
	}
	else {
		alt.levelType = LV_TYPE_NOT_DEFINED;
	}
	return alt;
}

//-------------------------------------------------
QString DataCodeStr::serialize (const DataCode &dtc)
{
	return QString("%1;%2;%3").arg(dtc.dataType).arg(dtc.levelType).arg(dtc.levelValue);
}
//-------------------------------------------------
DataCode DataCodeStr::unserialize (const QString &stringCode)
{
	DataCode dtc;
	QStringList list = stringCode.split(";");
	if (list.size() == 3) {
		bool ok;
		dtc.dataType = list.at(0).toInt (&ok);
		if (!ok) dtc.dataType = GRB_TYPE_NOT_DEFINED;
		dtc.levelType = list.at(1).toInt (&ok);
		if (!ok) dtc.dataType = GRB_TYPE_NOT_DEFINED;
		dtc.levelValue = list.at(2).toInt (&ok);
		if (!ok) dtc.dataType = GRB_TYPE_NOT_DEFINED;
	}
	else {
		dtc.dataType = GRB_TYPE_NOT_DEFINED;
	}
	return dtc;
}

