// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "components/password_manager/core/browser/store_metrics_reporter.h"

#include <algorithm>
#include <memory>
#include <string_view>
#include <utility>

#include "base/metrics/histogram_functions.h"
#include "base/notreached.h"
#include "base/strings/strcat.h"
#include "base/strings/utf_string_conversions.h"
#include "base/task/sequenced_task_runner.h"
#include "base/task/thread_pool.h"
#include "components/affiliations/core/browser/affiliation_utils.h"
#include "components/password_manager/core/browser/features/password_manager_features_util.h"
#include "components/password_manager/core/browser/password_feature_manager.h"
#include "components/password_manager/core/browser/password_form.h"
#include "components/password_manager/core/browser/password_manager_client.h"
#include "components/password_manager/core/browser/password_manager_metrics_util.h"
#include "components/password_manager/core/browser/password_manager_settings_service.h"
#include "components/password_manager/core/browser/password_reuse_detector.h"
#include "components/password_manager/core/browser/password_reuse_manager.h"
#include "components/password_manager/core/browser/password_store/password_store_consumer.h"
#include "components/password_manager/core/browser/password_store/password_store_interface.h"
#include "components/password_manager/core/browser/password_sync_util.h"
#include "components/password_manager/core/common/password_manager_pref_names.h"
#include "components/prefs/pref_service.h"
#include "components/safe_browsing/core/common/features.h"
#include "components/safe_browsing/core/common/safe_browsing_prefs.h"
#include "google_apis/gaia/gaia_auth_util.h"
#include "google_apis/gaia/gaia_urls.h"

namespace password_manager {

namespace {

// Minimum time between two metrics reporting trigger for the same profile. This
// is needed in order to avoid unnecessarily reporting password store metrics on
// every profile creation (which is very frequent on Android for example).
constexpr base::TimeDelta kMetricsReportingThreshold = base::Days(1);

// Common prefix for all histograms.
constexpr char kPasswordManager[] = "PasswordManager";

// Need to stay in sync with the PasswordType variant in histograms.xml.
constexpr char kAutoGeneratedSuffix[] = ".AutoGenerated";
constexpr char kUserCreatedSuffix[] = ".UserCreated";
constexpr char kReceivedViaSharingSuffix[] = ".ReceivedViaSharing";
constexpr char kImportedViaCredentialExchangeSuffix[] =
    ".ImportedViaCredentialExchange";
constexpr char kOverallSuffix[] = ".Overall";
constexpr char kExcludingStoreErrorsSuffix[] = ".ExcludingStoreErrors";

// Need to stay in sync with the CustomPassphraseStatus variant in
// histograms.xml.
constexpr char kWithCustomPassphraseSuffix[] = ".WithCustomPassphrase";
constexpr char kWithoutCustomPassphraseSuffix[] = ".WithoutCustomPassphrase";

// Suffix for the histogram that tracks password loss.
constexpr char kPasswordLossSuffix[] = ".PasswordLoss";

// Suffix for the histogram that tracks the reasons why passwords were recently
// removed.
constexpr char kPasswordLossPotentialReasonSuffix[] =
    ".PasswordLossPotentialReasonBitmask";

bool IsCustomPassphraseEnabled(
    password_manager::sync_util::SyncState sync_state) {
  switch (sync_state) {
    case password_manager::sync_util::SyncState::kNotActive:
    case password_manager::sync_util::SyncState::kActiveWithNormalEncryption:
      return false;
    case password_manager::sync_util::SyncState::kActiveWithCustomPassphrase:
      return true;
  }
  NOTREACHED();
}

std::string_view GetCustomPassphraseSuffix(bool custom_passphrase_enabled) {
  return custom_passphrase_enabled ? kWithCustomPassphraseSuffix
                                   : kWithoutCustomPassphraseSuffix;
}

// Returns a suffix (infix, really) to be used in histogram names to
// differentiate the profile store from the account store. Need to stay in sync
// with the Store variant in histograms.xml.
std::string_view GetMetricsSuffixForStore(bool is_account_store) {
  return is_account_store ? ".AccountStore" : ".ProfileStore";
}

// These values are persisted to logs. Entries should not be renumbered and
// numeric values should never be reused.
enum class SyncingAccountState {
  kSyncingAndSyncPasswordNotSaved = 0,
  kSyncingAndSyncPasswordSaved = 1,
  kNotSyncingAndSyncPasswordNotSaved = 2,
  kNotSyncingAndSyncPasswordSaved = 3,
  kMaxValue = kNotSyncingAndSyncPasswordSaved,
};

void LogAccountStatHiRes(const std::string& name, int sample) {
  base::UmaHistogramCustomCounts(name, sample, 0, 1000, 100);
}

void LogNumberOfAccountsForScheme(std::string_view suffix_for_store,
                                  const std::string& scheme,
                                  int sample) {
  base::UmaHistogramCustomCounts(
      base::StrCat({kPasswordManager, suffix_for_store,
                    ".TotalAccountsHiRes3.WithScheme.", scheme}),
      sample, 1, 1000, 100);
}

void LogTimesUsedStat(const std::string& name, int sample) {
  base::UmaHistogramCustomCounts(name, sample, 0, 100, 10);
}

int ReportNumberOfAccountsMetrics(bool is_account_store,
                                  bool custom_passphrase_enabled,
                                  const PasswordStoreResults& results) {
  base::flat_map<std::tuple<std::string, PasswordForm::Type, int>, int>
      accounts_per_site_map;

  for (const auto& form : results.store_results) {
    accounts_per_site_map[{form->signon_realm, form->type,
                           form->blocked_by_user}]++;
  }

  std::string_view store_suffix = GetMetricsSuffixForStore(is_account_store);
  std::string_view custom_passphrase_suffix =
      GetCustomPassphraseSuffix(custom_passphrase_enabled);

  int total_user_created_accounts = 0;
  int total_generated_accounts = 0;
  int total_received_via_sharing_accounts = 0;
  int total_imported_via_credential_exchange = 0;
  int blocklisted_sites = 0;
  for (const auto& pair : accounts_per_site_map) {
    PasswordForm::Type password_type = std::get<1>(pair.first);
    int blocklisted = std::get<2>(pair.first);
    int accounts_per_site = pair.second;
    if (blocklisted) {
      ++blocklisted_sites;
      continue;
    }

    constexpr std::string_view kAccountsPerSiteSuffix =
        ".AccountsPerSiteHiRes3";

    if (password_type == PasswordForm::Type::kGenerated) {
      total_generated_accounts += accounts_per_site;
      LogAccountStatHiRes(
          base::StrCat({kPasswordManager, store_suffix, kAccountsPerSiteSuffix,
                        kAutoGeneratedSuffix, custom_passphrase_suffix}),
          accounts_per_site);
    } else if (password_type == PasswordForm::Type::kReceivedViaSharing) {
      total_received_via_sharing_accounts += accounts_per_site;
      LogAccountStatHiRes(
          base::StrCat({kPasswordManager, store_suffix, kAccountsPerSiteSuffix,
                        kReceivedViaSharingSuffix, custom_passphrase_suffix}),
          accounts_per_site);
    } else if (password_type ==
               PasswordForm::Type::kImportedViaCredentialExchange) {
      total_imported_via_credential_exchange += accounts_per_site;
      LogAccountStatHiRes(
          base::StrCat({kPasswordManager, store_suffix, kAccountsPerSiteSuffix,
                        kImportedViaCredentialExchangeSuffix,
                        custom_passphrase_suffix}),
          accounts_per_site);
    } else {
      total_user_created_accounts += accounts_per_site;
      LogAccountStatHiRes(
          base::StrCat({kPasswordManager, store_suffix, kAccountsPerSiteSuffix,
                        kUserCreatedSuffix, custom_passphrase_suffix}),
          accounts_per_site);
    }

    LogAccountStatHiRes(
        base::StrCat({kPasswordManager, store_suffix, kAccountsPerSiteSuffix,
                      kOverallSuffix, custom_passphrase_suffix}),
        accounts_per_site);

    // Same as above but not split by custom passphrase.
    LogAccountStatHiRes(base::StrCat({kPasswordManager, store_suffix,
                                      kAccountsPerSiteSuffix, kOverallSuffix}),
                        accounts_per_site);
  }

  static constexpr std::string_view kTotalAccountsByTypeSuffix =
      ".TotalAccountsHiRes3.ByType";

  LogAccountStatHiRes(
      base::StrCat({kPasswordManager, store_suffix, kTotalAccountsByTypeSuffix,
                    kUserCreatedSuffix, custom_passphrase_suffix}),
      total_user_created_accounts);

  LogAccountStatHiRes(
      base::StrCat({kPasswordManager, store_suffix, kTotalAccountsByTypeSuffix,
                    kAutoGeneratedSuffix, custom_passphrase_suffix}),
      total_generated_accounts);

  LogAccountStatHiRes(
      base::StrCat({kPasswordManager, store_suffix, kTotalAccountsByTypeSuffix,
                    kReceivedViaSharingSuffix, custom_passphrase_suffix}),
      total_received_via_sharing_accounts);

  LogAccountStatHiRes(
      base::StrCat({kPasswordManager, store_suffix, kTotalAccountsByTypeSuffix,
                    kImportedViaCredentialExchangeSuffix,
                    custom_passphrase_suffix}),
      total_imported_via_credential_exchange);

  int total_accounts = total_user_created_accounts + total_generated_accounts +
                       total_received_via_sharing_accounts +
                       total_imported_via_credential_exchange;
  LogAccountStatHiRes(
      base::StrCat({kPasswordManager, store_suffix, kTotalAccountsByTypeSuffix,
                    kOverallSuffix, custom_passphrase_suffix}),
      total_accounts);

  // Same as above but not split by custom passphrase which is most always
  // useless.
  LogAccountStatHiRes(
      base::StrCat({kPasswordManager, store_suffix, kTotalAccountsByTypeSuffix,
                    kOverallSuffix}),
      total_accounts);

  if (!results.has_error) {
    LogAccountStatHiRes(
        base::StrCat({kPasswordManager, store_suffix,
                      kTotalAccountsByTypeSuffix, kOverallSuffix,
                      kExcludingStoreErrorsSuffix}),
        total_accounts);
  }

  LogAccountStatHiRes(
      base::StrCat({kPasswordManager, store_suffix, ".BlacklistedSitesHiRes3",
                    custom_passphrase_suffix}),
      blocklisted_sites);
  return total_accounts;
}

void ReportLoginsWithSchemesMetrics(
    bool is_account_store,
    const std::vector<std::unique_ptr<PasswordForm>>& forms) {
  int android_logins = 0;
  int ftp_logins = 0;
  int http_logins = 0;
  int https_logins = 0;
  int other_logins = 0;

  for (const auto& form : forms) {
    if (form->blocked_by_user) {
      continue;
    }

    if (affiliations::IsValidAndroidFacetURI(form->signon_realm)) {
      ++android_logins;
    } else if (form->url.SchemeIs(url::kHttpsScheme)) {
      ++https_logins;
    } else if (form->url.SchemeIs(url::kHttpScheme)) {
      ++http_logins;
    } else if (form->url.SchemeIs(url::kFtpScheme)) {
      ++ftp_logins;
    } else {
      ++other_logins;
    }
  }

  std::string_view suffix_for_store =
      GetMetricsSuffixForStore(is_account_store);

  LogNumberOfAccountsForScheme(suffix_for_store, "Android", android_logins);
  LogNumberOfAccountsForScheme(suffix_for_store, "Ftp", ftp_logins);
  LogNumberOfAccountsForScheme(suffix_for_store, "Http", http_logins);
  LogNumberOfAccountsForScheme(suffix_for_store, "Https", https_logins);
  LogNumberOfAccountsForScheme(suffix_for_store, "Other", other_logins);
}

// These values are persisted to logs. Entries should not be renumbered and
// numeric values should never be reused.
enum class PasswordManagerEnableState {
  kDefault = 0,
  kEnabledByUser = 1,
  kEnabledByExtension = 2,
  kEnabledByPolicy = 3,
  kEnabledByRecommendedPolicy = 4,
  kEnabledByOther = 5,
  kDisabledByUser = 6,
  kDisabledByExtension = 7,
  kDisabledByPolicy = 8,
  kDisabledByRecommendedPolicy = 9,
  kDisabledByOther = 10,
  kMaxValue = kDisabledByOther,
};

PasswordManagerEnableState
CredentialsEnableServiceSettingToPasswordManagerEnableState(
    const PrefService::Preference* pref) {
  DCHECK(pref->GetValue()->GetIfBool().has_value());

  if (pref->IsDefaultValue()) {
    return PasswordManagerEnableState::kDefault;
  }

  bool credentials_service_enabled = pref->GetValue()->GetBool();
  if (pref->IsUserControlled()) {
    return credentials_service_enabled
               ? PasswordManagerEnableState::kEnabledByUser
               : PasswordManagerEnableState::kDisabledByUser;
  }
  if (pref->IsManaged()) {
    return credentials_service_enabled
               ? PasswordManagerEnableState::kEnabledByPolicy
               : PasswordManagerEnableState::kDisabledByPolicy;
  }
  if (pref->IsExtensionControlled()) {
    return credentials_service_enabled
               ? PasswordManagerEnableState::kEnabledByExtension
               : PasswordManagerEnableState::kDisabledByExtension;
  }
  if (pref->IsRecommended()) {
    return credentials_service_enabled
               ? PasswordManagerEnableState::kEnabledByRecommendedPolicy
               : PasswordManagerEnableState::kDisabledByRecommendedPolicy;
  }
  return credentials_service_enabled
             ? PasswordManagerEnableState::kEnabledByOther
             : PasswordManagerEnableState::kDisabledByOther;
}

void ReportPasswordNotesMetrics(
    bool is_account_store,
    const std::vector<std::unique_ptr<PasswordForm>>& forms) {
  std::string_view suffix_for_store =
      GetMetricsSuffixForStore(is_account_store);

  int credentials_with_non_empty_notes_count =
      std::ranges::count_if(forms, [](const auto& form) {
        return std::ranges::any_of(
            form->notes, [](const auto& note) { return !note.value.empty(); });
      });

  base::UmaHistogramCounts1000(
      base::StrCat({kPasswordManager, suffix_for_store,
                    ".PasswordNotes.CountCredentialsWithNonEmptyNotes2"}),
      credentials_with_non_empty_notes_count);

  const std::string histogram_name =
      base::StrCat({kPasswordManager, suffix_for_store,
                    ".PasswordNotes.CountNotesPerCredential3"});
  std::ranges::for_each(forms, [histogram_name](const auto& form) {
    if (!form->notes.empty()) {
      base::UmaHistogramCounts100(histogram_name, form->notes.size());
    }
  });
}

void ReportTimesPasswordUsedMetrics(
    bool is_account_store,
    bool custom_passphrase_enabled,
    const std::vector<std::unique_ptr<PasswordForm>>& forms) {
  std::string_view store_suffix = GetMetricsSuffixForStore(is_account_store);
  std::string_view custom_passphrase_suffix =
      GetCustomPassphraseSuffix(custom_passphrase_enabled);

  for (const auto& form : forms) {
    auto type = form->type;
    const int times_used_in_html_form = form->times_used_in_html_form;

    static constexpr std::string_view kTimesPasswordUsedSuffix =
        ".TimesPasswordUsed3";

    if (type == PasswordForm::Type::kGenerated) {
      LogTimesUsedStat(
          base::StrCat({kPasswordManager, store_suffix,
                        kTimesPasswordUsedSuffix, kAutoGeneratedSuffix,
                        custom_passphrase_suffix}),
          times_used_in_html_form);
    } else if (type == PasswordForm::Type::kReceivedViaSharing) {
      LogTimesUsedStat(
          base::StrCat({kPasswordManager, store_suffix,
                        kTimesPasswordUsedSuffix, kReceivedViaSharingSuffix,
                        custom_passphrase_suffix}),
          times_used_in_html_form);
    } else if (type == PasswordForm::Type::kImportedViaCredentialExchange) {
      LogTimesUsedStat(
          base::StrCat(
              {kPasswordManager, store_suffix, kTimesPasswordUsedSuffix,
               kImportedViaCredentialExchangeSuffix, custom_passphrase_suffix}),
          times_used_in_html_form);
    } else {
      LogTimesUsedStat(
          base::StrCat({kPasswordManager, store_suffix,
                        kTimesPasswordUsedSuffix, kUserCreatedSuffix,
                        custom_passphrase_suffix}),
          times_used_in_html_form);
    }
    LogTimesUsedStat(
        base::StrCat({kPasswordManager, store_suffix, kTimesPasswordUsedSuffix,
                      kOverallSuffix, custom_passphrase_suffix}),
        times_used_in_html_form);
  }
}

void ReportSyncingAccountStateMetrics(
    const std::string& sync_username,
    const std::vector<std::unique_ptr<PasswordForm>>& forms) {
  const GURL gaia_signon_realm =
      GaiaUrls::GetInstance()->gaia_origin().GetURL();
  bool syncing_account_saved = std::ranges::any_of(
      forms, [&gaia_signon_realm, &sync_username](const auto& form) {
        return gaia_signon_realm == GURL(form->signon_realm) &&
               gaia::AreEmailsSame(sync_username,
                                   base::UTF16ToUTF8(form->username_value));
      });
  SyncingAccountState sync_account_state =
      sync_username.empty()
          ? (syncing_account_saved
                 ? SyncingAccountState::kNotSyncingAndSyncPasswordSaved
                 : SyncingAccountState::kNotSyncingAndSyncPasswordNotSaved)
          : (syncing_account_saved
                 ? SyncingAccountState::kSyncingAndSyncPasswordSaved
                 : SyncingAccountState::kSyncingAndSyncPasswordNotSaved);
  base::UmaHistogramEnumeration(
      base::StrCat({kPasswordManager, ".SyncingAccountState3"}),
      sync_account_state);
}

void ReportDuplicateCredentialsMetrics(
    const std::vector<std::unique_ptr<PasswordForm>>& forms) {
  // First group the passwords by [signon_realm, username] (which should be a
  // unique identifier).
  std::map<std::pair<std::string, std::u16string>, std::vector<std::u16string>>
      passwords_by_realm_and_user;
  for (const auto& form : forms) {
    passwords_by_realm_and_user[std::make_pair(form->signon_realm,
                                               form->username_value)]
        .push_back(form->password_value);
  }
  // Now go over the passwords by [realm, username] - typically there should
  // be only one password each.
  size_t credentials_with_duplicates = 0;
  size_t credentials_with_mismatched_duplicates = 0;
  for (auto& entry : passwords_by_realm_and_user) {
    std::vector<std::u16string>& passwords = entry.second;
    // Only one password -> no duplicates, move on.
    if (passwords.size() == 1) {
      continue;
    }
    std::sort(passwords.begin(), passwords.end());
    auto last = std::unique(passwords.begin(), passwords.end());
    // If |last| moved from |.end()|, that means there were duplicate
    // passwords.
    if (last != passwords.end()) {
      credentials_with_duplicates++;
    }
    // If there is more than 1 password left after de-duping, then there were
    // mismatched duplicates.
    if (std::distance(passwords.begin(), last) > 1) {
      credentials_with_mismatched_duplicates++;
    }
  }

  base::UmaHistogramCustomCounts(
      base::StrCat({kPasswordManager, ".CredentialsWithDuplicates3"}),
      credentials_with_duplicates, 0, 32, 6);
  base::UmaHistogramCustomCounts(
      base::StrCat({kPasswordManager, ".CredentialsWithMismatchedDuplicates3"}),
      credentials_with_mismatched_duplicates, 0, 32, 6);
}

void ReportPasswordIssuesMetrics(
    const std::vector<std::unique_ptr<PasswordForm>>& forms) {
  int count_leaked = std::ranges::count_if(forms, [](const auto& form) {
    return form->password_issues.contains(InsecureType::kLeaked);
  });
  base::UmaHistogramCounts100(
      base::StrCat({kPasswordManager, ".CompromisedCredentials3.CountLeaked"}),
      count_leaked);

  int count_phished = std::ranges::count_if(forms, [](const auto& form) {
    return form->password_issues.contains(InsecureType::kPhished);
  });
  base::UmaHistogramCounts100(
      base::StrCat({kPasswordManager, ".CompromisedCredentials3.CountPhished"}),
      count_phished);
}

void ReportPasswordProtectedMetrics(
    const std::vector<std::unique_ptr<PasswordForm>>& forms) {
  for (const std::unique_ptr<PasswordForm>& form : forms) {
    if (!form->blocked_by_user && form->password_value.size() > 0) {
      metrics_util::LogIsPasswordProtected(form->password_value.size() >=
                                           kMinPasswordLengthToCheck);
    }
  }
}

int ReportStoreMetrics(bool is_account_store,
                       bool custom_passphrase_enabled,
                       const std::string& sync_username,
                       bool is_safe_browsing_enabled,
                       PasswordStoreResults password_store_results) {
  std::vector<std::unique_ptr<PasswordForm>>& results =
      password_store_results.store_results;

  int total_accounts = ReportNumberOfAccountsMetrics(
      is_account_store, custom_passphrase_enabled, password_store_results);
  ReportLoginsWithSchemesMetrics(is_account_store, results);
  ReportTimesPasswordUsedMetrics(is_account_store, custom_passphrase_enabled,
                                 results);
  ReportPasswordNotesMetrics(is_account_store, results);
  if (is_safe_browsing_enabled) {
    ReportPasswordProtectedMetrics(results);
  }

  // The remaining metrics are not recorded for the account store:
  // - SyncingAccountState2 just doesn't make sense, since syncing users only
  // use
  //   the profile store.
  // - DuplicateCredentials *could* be recorded for the account store, but are
  //   not very critical.
  // - Compromised credentials are only stored in the profile store.
  // TODO: crbug.com/344573277 - Rethink about the comment above. The note about
  // SyncingAccountState2 is not true on Android and the one about
  // DuplicateCredentials may change in the future.
  if (is_account_store) {
    return total_accounts;
  }

  ReportSyncingAccountStateMetrics(sync_username, results);
  ReportDuplicateCredentialsMetrics(results);
  ReportPasswordIssuesMetrics(results);
  return total_accounts;
}

void ReportMultiStoreMetrics(
    std::unique_ptr<std::map<std::pair<std::string, std::u16string>,
                             std::u16string>> profile_store_results,
    std::unique_ptr<std::map<std::pair<std::string, std::u16string>,
                             std::u16string>> account_store_results,
    bool is_account_storage_enabled) {
  // Count the contents of the account store as compared to the profile store:
  // - Additional:  Credentials that are in the account store, but not in the
  //                profile store.
  // - Missing:     Credentials that are in the profile store, but not in the
  //                account store.
  // - Identical:   Credentials that are in both stores.
  // - Conflicting: Credentials with the same signon realm and username, but
  //                different passwords in the two stores.
  int additional = 0;
  int missing = 0;
  int identical = 0;
  int conflicting = 0;

  // Go over the data from both stores in parallel, always advancing in the
  // one that is "behind". The entries are sorted by signon_realm and
  // username (the exact ordering doesn't matter, just that it's consistent).
  auto profile_it = profile_store_results->begin();
  auto account_it = account_store_results->begin();
  while (account_it != account_store_results->end()) {
    // First, go over any entries in the profile store that don't exist in the
    // account store.
    while (profile_it != profile_store_results->end() &&
           profile_it->first < account_it->first) {
      ++missing;
      ++profile_it;
    }
    // Now profile_it->first is >= account_it->first.
    // Check if they match.
    if (profile_it != profile_store_results->end() &&
        account_it->first == profile_it->first) {
      // The signon_realm and username match, check the password value.
      if (account_it->second == profile_it->second) {
        ++identical;
      } else {
        ++conflicting;
      }

      ++profile_it;
    } else {
      // The signon_realm and username don't match, so this is an account
      // store entry that doesn't exist in the profile store.
      ++additional;
    }

    ++account_it;
  }
  // We're done with the account store. Go over any remaining profile store
  // entries.
  while (profile_it != profile_store_results->end()) {
    ++missing;
    ++profile_it;
  }

  if (is_account_storage_enabled) {
    base::UmaHistogramCounts100(
        base::StrCat(
            {kPasswordManager, ".AccountStoreVsProfileStore4.Additional"}),
        additional);
    base::UmaHistogramCounts100(
        base::StrCat(
            {kPasswordManager, ".AccountStoreVsProfileStore4.Missing"}),
        missing);
    base::UmaHistogramCounts100(
        base::StrCat(
            {kPasswordManager, ".AccountStoreVsProfileStore4.Identical"}),
        identical);
    base::UmaHistogramCounts100(
        base::StrCat(
            {kPasswordManager, ".AccountStoreVsProfileStore4.Conflicting"}),
        conflicting);
  }
}

StoreMetricsReporter::CredentialsCount ReportAllMetrics(
    bool custom_passphrase_enabled,
    const std::string& sync_username,
    bool is_account_storage_enabled,
    bool is_safe_browsing_enabled,
    std::optional<PasswordStoreResults> profile_store_results,
    std::optional<PasswordStoreResults> account_store_results) {
  // Maps from (signon_realm, username) to password.
  std::unique_ptr<
      std::map<std::pair<std::string, std::u16string>, std::u16string>>
      profile_store_passwords_per_signon_and_username;
  std::unique_ptr<
      std::map<std::pair<std::string, std::u16string>, std::u16string>>
      account_store_passwords_per_signon_and_username;

  if (profile_store_results.has_value()) {
    profile_store_passwords_per_signon_and_username = std::make_unique<
        std::map<std::pair<std::string, std::u16string>, std::u16string>>();
    for (const std::unique_ptr<PasswordForm>& form :
         profile_store_results.value().store_results) {
      profile_store_passwords_per_signon_and_username->insert(std::make_pair(
          std::make_pair(form->signon_realm, form->username_value),
          form->password_value));
    }
  }

  if (account_store_results.has_value()) {
    account_store_passwords_per_signon_and_username = std::make_unique<
        std::map<std::pair<std::string, std::u16string>, std::u16string>>();
    for (const std::unique_ptr<PasswordForm>& form :
         account_store_results.value().store_results) {
      account_store_passwords_per_signon_and_username->insert(std::make_pair(
          std::make_pair(form->signon_realm, form->username_value),
          form->password_value));
    }
  }

  StoreMetricsReporter::CredentialsCount credentials_count;

  if (profile_store_results.has_value()) {
    credentials_count.profile_credentials_count = ReportStoreMetrics(
        /*is_account_store=*/false, custom_passphrase_enabled, sync_username,
        is_safe_browsing_enabled, std::move(profile_store_results).value());
  }
  if (account_store_results.has_value()) {
    credentials_count.account_credentials_count = ReportStoreMetrics(
        /*is_account_store=*/true, custom_passphrase_enabled, sync_username,
        is_safe_browsing_enabled, std::move(account_store_results).value());
  }

  // If both stores exist, kick off the MultiStoreMetricsReporter.
  if (profile_store_passwords_per_signon_and_username &&
      account_store_passwords_per_signon_and_username) {
    ReportMultiStoreMetrics(
        std::move(profile_store_passwords_per_signon_and_username),
        std::move(account_store_passwords_per_signon_and_username),
        is_account_storage_enabled);
  }

  return credentials_count;
}

void ReportBiometricAuthenticationBeforeFillingMetrics(PrefService* prefs) {
#if BUILDFLAG(IS_MAC) || BUILDFLAG(IS_WIN) || BUILDFLAG(IS_CHROMEOS)
  base::UmaHistogramBoolean(
      base::StrCat({kPasswordManager, ".BiometricAuthBeforeFillingEnabled2"}),
      prefs->GetBoolean(
          password_manager::prefs::kBiometricAuthenticationBeforeFilling));
#endif
}

void ReportPasswordReencryption(PrefService* prefs) {
  constexpr std::string_view kName = ".ReencryptedWithAsyncOSCrypt";
  base::UmaHistogramBoolean(
      base::StrCat({kPasswordManager,
                    GetMetricsSuffixForStore(/*is_account_store=*/false),
                    kName}),
      prefs->GetBoolean(prefs::kProfileStoreMigratedToOSCryptAsync));
  base::UmaHistogramBoolean(
      base::StrCat({kPasswordManager,
                    GetMetricsSuffixForStore(/*is_account_store=*/true),
                    kName}),
      prefs->GetBoolean(prefs::kAccountStoreMigratedToOSCryptAsync));
}

}  // namespace

PasswordStoreResults::PasswordStoreResults(
    std::vector<std::unique_ptr<PasswordForm>> store_results,
    bool has_error)
    : store_results(std::move(store_results)), has_error(has_error) {}
PasswordStoreResults::~PasswordStoreResults() = default;
PasswordStoreResults::PasswordStoreResults(PasswordStoreResults&& other) =
    default;
PasswordStoreResults& PasswordStoreResults::operator=(
    PasswordStoreResults&& other) = default;

StoreMetricsReporter::StoreMetricsReporter(
    PasswordStoreInterface* profile_store,
    PasswordStoreInterface* account_store,
    const syncer::SyncService* sync_service,
    PrefService* prefs,
    password_manager::PasswordReuseManager* password_reuse_manager,
    PasswordManagerSettingsService* settings,
    base::OnceClosure done_callback)
    : profile_store_(profile_store),
      account_store_(account_store),
      prefs_(prefs),
      done_callback_(std::move(done_callback)) {
  base::TimeDelta time_since_last_metrics_reporting =
      base::Time::Now() -
      base::Time::FromTimeT(prefs_->GetDouble(
          password_manager::prefs::kLastTimePasswordStoreMetricsReported));
  if (time_since_last_metrics_reporting < kMetricsReportingThreshold) {
    // Upon constructing StoreMetricsReporter, it's moved into member variable
    // in StoreMetricReporterHelper. `done_callback_` effectively destroys
    // StoreMetricReporterHelper. Therefore, `done_callback_` must be called
    // asynchronously to avoid moving the StoreMetricsReporter pointer to a
    // destroyed unique_ptr.
    base::SequencedTaskRunner::GetCurrentDefault()->PostTask(
        FROM_HERE, std::move(done_callback_));
    return;
  }

  prefs_->SetDouble(
      password_manager::prefs::kLastTimePasswordStoreMetricsReported,
      base::Time::Now().InSecondsFSinceUnixEpoch());

  sync_username_ = password_manager::sync_util::
      GetAccountEmailIfSyncFeatureEnabledIncludingPasswords(sync_service);

  custom_passphrase_enabled_ = IsCustomPassphraseEnabled(
      password_manager::sync_util::GetPasswordSyncState(sync_service));

  is_account_storage_enabled_ =
      features_util::IsAccountStorageEnabled(sync_service);

  is_safe_browsing_enabled_ = safe_browsing::IsSafeBrowsingEnabled(*prefs_);

  if (settings) {
    // TODO(crbug.com/358998546): use PasswordManagerSettingsService here.
    base::UmaHistogramEnumeration(
        base::StrCat({kPasswordManager, ".EnableState"}),
        CredentialsEnableServiceSettingToPasswordManagerEnableState(
            prefs_->FindPreference(
                password_manager::prefs::kCredentialsEnableService)));
    base::UmaHistogramBoolean(
        base::StrCat({kPasswordManager, ".AutoSignin"}),
        settings->IsSettingEnabled(PasswordManagerSetting::kAutoSignIn));
  }

  ReportBiometricAuthenticationBeforeFillingMetrics(prefs_);
  ReportPasswordReencryption(prefs_);

  // May be null in tests.
  if (profile_store) {
    if (password_reuse_manager) {
      password_reuse_manager->ReportMetrics(sync_username_);
    }
  }

  if (profile_store_) {
    profile_store_->GetAllLogins(weak_ptr_factory_.GetWeakPtr());
  }

  if (account_store_) {
    account_store_->GetAllLogins(weak_ptr_factory_.GetWeakPtr());
  }

  if (!profile_store_ && !account_store_) {
    // There is nothing else to report.
    base::SequencedTaskRunner::GetCurrentDefault()->PostTask(
        FROM_HERE, std::move(done_callback_));
  }
}

void StoreMetricsReporter::OnGetPasswordStoreResults(
    std::vector<std::unique_ptr<PasswordForm>> results) {
  // This class overrides OnGetPasswordStoreResultsFrom() (the version of this
  // method that also receives the originating store), so the store-less version
  // never gets called.
  NOTREACHED();
}

void StoreMetricsReporter::OnGetPasswordStoreResultsFrom(
    PasswordStoreInterface* store,
    std::vector<std::unique_ptr<PasswordForm>> results) {
  // This class overrides OnGetPasswordStoreResultsOrErrorFrom() (the version
  // that also receives the error case), so the plain password form version
  // never gets called.
  NOTREACHED();
}

void StoreMetricsReporter::OnGetPasswordStoreResultsOrErrorFrom(
    PasswordStoreInterface* store,
    LoginsResultOrError results_or_error) {
  PasswordStoreResults password_store_results{
      password_manager::ConvertPasswordToUniquePtr(
          password_manager::GetLoginsOrEmptyListOnFailure(
              std::move(results_or_error))),
      std::holds_alternative<PasswordStoreBackendError>(results_or_error)};

  ProcessPasswordResults(store, std::move(password_store_results));
}

void StoreMetricsReporter::ProcessPasswordResults(
    PasswordStoreInterface* store,
    PasswordStoreResults results) {
  if (store == account_store_) {
    account_store_results_ = std::move(results);
  } else {
    profile_store_results_ = std::move(results);
  }

  // Wait until all expected results are available before starting metrics
  // reporting.
  if ((profile_store_ && !profile_store_results_) ||
      (account_store_ && !account_store_results_)) {
    return;
  }

  DCHECK(done_callback_);

  // Metrics reporting is performed asynchronously on a background thread. By
  // the time metrics reporting is completed, it could be the case that the
  // StoreMetricsReporter has been destructed already (e.g. if the user closes
  // the browser profile for which metrics are being reported) so
  // `OnBackgroundMetricsReportingCompleted` won't run.
  base::ThreadPool::PostTaskAndReplyWithResult(
      FROM_HERE, {base::TaskPriority::BEST_EFFORT, base::MayBlock()},
      base::BindOnce(&ReportAllMetrics, custom_passphrase_enabled_,
                     sync_username_, is_account_storage_enabled_,
                     is_safe_browsing_enabled_,
                     std::exchange(profile_store_results_, std::nullopt),
                     std::exchange(account_store_results_, std::nullopt)),
      base::BindOnce(
          &StoreMetricsReporter::OnBackgroundMetricsReportingCompleted,
          weak_ptr_factory_.GetWeakPtr()));
}

StoreMetricsReporter::~StoreMetricsReporter() {
  // Avoid complaints in case those objects are already dead.
  prefs_ = nullptr;
}

void StoreMetricsReporter::OnBackgroundMetricsReportingCompleted(
    CredentialsCount credentials_count) {
  // Check for password loss and record metrics if a loss occurred.
  // These metrics can't be recorded together with the rest of the metrics on
  // the background thread because they require reading from Chrome prefs,
  // which can't happen on the background thread.
  int old_account_credentials_count =
      prefs_->GetInteger(prefs::kTotalPasswordsAvailableForAccount);
  if (old_account_credentials_count > 0 &&
      credentials_count.account_credentials_count == 0) {
    std::string_view store_suffix =
        GetMetricsSuffixForStore(/*is_account_store=*/true);
    base::UmaHistogramCustomCounts(
        base::StrCat({kPasswordManager, store_suffix, kPasswordLossSuffix}),
        old_account_credentials_count, 0, 1000, 100);

    int credential_removal_reasons =
        prefs_->GetInteger(prefs::kPasswordRemovalReasonForAccount);
    base::UmaHistogramSparse(base::StrCat({kPasswordManager, store_suffix,
                                           kPasswordLossPotentialReasonSuffix}),
                             credential_removal_reasons);
  }

  int old_profile_credentials_count =
      prefs_->GetInteger(prefs::kTotalPasswordsAvailableForProfile);
  if (old_profile_credentials_count > 0 &&
      credentials_count.profile_credentials_count == 0) {
    std::string_view store_suffix =
        GetMetricsSuffixForStore(/*is_account_store=*/false);
    base::UmaHistogramCustomCounts(
        base::StrCat({kPasswordManager, store_suffix, kPasswordLossSuffix}),
        old_profile_credentials_count, 0, 1000, 100);

    int credential_removal_reasons =
        prefs_->GetInteger(prefs::kPasswordRemovalReasonForProfile);
    base::UmaHistogramSparse(base::StrCat({kPasswordManager, store_suffix,
                                           kPasswordLossPotentialReasonSuffix}),
                             credential_removal_reasons);
  }

  // Store the current total count of passwords per store for tracking
  // potential password loss in the future.
  prefs_->SetInteger(prefs::kTotalPasswordsAvailableForAccount,
                     credentials_count.account_credentials_count);
  prefs_->SetInteger(prefs::kTotalPasswordsAvailableForProfile,
                     credentials_count.profile_credentials_count);

  // The reasons for password loss need to be tracked anew because the old
  // ones were already processed.
  prefs_->ClearPref(prefs::kPasswordRemovalReasonForAccount);
  prefs_->ClearPref(prefs::kPasswordRemovalReasonForProfile);

  // `done_callback_` may delete `this` object.
  std::move(done_callback_).Run();
}

}  // namespace password_manager
